import ctypes
import time
import os
import sys
from PyQt5.QtCore import QObject, pyqtSignal
from yc.log_manager import LogManager


# ===== VLC路径设置（迁移自主程序）=====
def setup_vlc_path():
    """在导入vlc模块之前设置DLL路径"""
    # 检查是否在打包环境中运行
    if getattr(sys, 'frozen', False):
        # 打包环境 - 优先使用_internal\vlc目录
        base_path = sys._MEIPASS if hasattr(sys, '_MEIPASS') else os.path.dirname(sys.executable)
        # 尝试多个可能的VLC路径位置
        vlc_paths = [
            os.path.join(base_path, 'vlc'),  # 备用路径
            os.path.join(base_path, '_internal', 'vlc'),  # 主路径
            os.path.join(os.path.dirname(sys.executable), '_internal', 'vlc'),  # 相对于可执行文件
        ]
        LogManager().debug(f"程序运行路径: {base_path}")
        LogManager().debug("尝试查找VLC路径:")
        for vlc_path in vlc_paths:
            LogManager().debug(f"  检查路径: {vlc_path}")
            if os.path.exists(vlc_path):
                # 检查是否存在libvlc.dll
                libvlc_path = os.path.join(vlc_path, "libvlc.dll")
                if os.path.exists(libvlc_path):
                    LogManager().debug(f"  找到VLC路径: {vlc_path}")
                    # 设置环境变量
                    os.environ['VLC_PLUGIN_PATH'] = vlc_path
                    # 将VLC目录添加到DLL搜索路径（关键步骤）
                    if hasattr(os, 'add_dll_directory'):
                        try:
                            os.add_dll_directory(vlc_path)
                            LogManager().debug(f"  已添加VLC路径到DLL搜索目录: {vlc_path}")
                        except Exception as e:
                            LogManager().debug(f"  添加DLL搜索目录失败: {e}")
                    # 尝试直接预加载libvlc.dll
                    try:
                        ctypes.CDLL(libvlc_path)
                        LogManager().debug(f"  成功预加载VLC DLL: {libvlc_path}")
                        return vlc_path
                    except Exception as e:
                        LogManager().debug(f"  预加载VLC DLL失败: {e}")
                else:
                    LogManager().debug(f"  路径存在但缺少libvlc.dll: {libvlc_path}")
            else:
                LogManager().debug(f"  路径不存在: {vlc_path}")
        LogManager().debug("警告: 在打包环境中未找到VLC目录")
    # 开发环境或打包环境未找到VLC
    # 尝试从系统查找VLC
    possible_paths = [
        os.path.join(os.environ.get("PROGRAMFILES", "C:\\Program Files"), "VideoLAN", "VLC"),
        os.path.join(os.environ.get("PROGRAMFILES(X86)", "C:\\Program Files (x86)"), "VideoLAN", "VLC"),
        "C:\\Program Files\\VideoLAN\\VLC",
        "C:\\Program Files (x86)\\VideoLAN\\VLC",
        os.path.expanduser("~/AppData/Local/Programs/VideoLAN/VLC")
    ]
    LogManager().debug("尝试查找系统VLC路径:")
    for path in possible_paths:
        LogManager().debug(f"  检查路径: {path}")
        if os.path.exists(os.path.join(path, "libvlc.dll")):
            LogManager().debug(f"  使用系统VLC路径: {path}")
            os.environ['VLC_PLUGIN_PATH'] = path
            if hasattr(os, 'add_dll_directory'):
                try:
                    os.add_dll_directory(path)
                except Exception as e:
                    LogManager().debug(f"  添加系统VLC路径到DLL搜索目录失败: {e}")
            return path
    LogManager().debug("警告: 未找到任何VLC安装")
    return None


# 自动执行VLC路径设置
vlc_path = setup_vlc_path()
if vlc_path:
    LogManager().debug(f"使用VLC路径: {vlc_path}")
    # 确保VLC路径在系统路径中
    if vlc_path not in os.environ.get('PATH', ''):
        os.environ['PATH'] = vlc_path + os.pathsep + os.environ.get('PATH', '')
        LogManager().debug(f"已将VLC路径添加到系统PATH: {vlc_path}")

# 安全导入VLC
try:
    import vlc

    VLC_AVAILABLE = True
    LogManager().info("VLC模块导入成功")
except Exception as e:
    LogManager().debug(f"VLC模块导入失败: {e}")
    vlc = None
    VLC_AVAILABLE = False


class MusicPlayer(QObject):
    """音乐播放控制管理器"""
    # 定义信号
    track_ended = pyqtSignal()  # 播放结束信号
    position_changed = pyqtSignal(int)  # 位置变化信号（毫秒）
    volume_changed = pyqtSignal(int)  # 音量变化信号
    playback_state_changed = pyqtSignal(bool, bool)  # 播放状态变化信号 (playing, paused)
    duration_changed = pyqtSignal(int)  # 时长变化信号（单位：毫秒）

    def __init__(self):
        super().__init__()
        self.logger = LogManager()
        # VLC相关属性
        self.vlc_instance = None
        self.vlc_player = None
        self.vlc_events = None
        # 播放状态
        self.playing = False
        self.paused = False
        self.current_track = None
        self.position = 0
        self.duration = 0
        self.volume = 80
        self.logger.debug("MusicPlayer初始化完成")

    def init_vlc(self):
        """初始化VLC播放器"""
        if not VLC_AVAILABLE:
            self.logger.debug("VLC模块不可用，初始化失败")
            return False
        try:
            self.logger.debug("开始初始化VLC")
            # 创建VLC实例，移除已弃用的参数
            vlc_args = [
                '--no-xlib',  # 禁用X11支持
                '--no-stats',  # 禁用统计信息
                '--no-osd',  # 禁用屏幕显示
                '--quiet',  # 静默模式
                '--no-video-title-show',  # 不显示视频标题
                '--no-sub-autodetect-file',  # 不自动检测字幕文件
            ]
            self.vlc_instance = vlc.Instance(vlc_args)
            self.vlc_player = self.vlc_instance.media_player_new()
            self.vlc_events = self.vlc_player.event_manager()
            # 只保留播放结束事件监听
            self.vlc_events.event_attach(vlc.EventType.MediaPlayerEndReached, self._on_track_end)
            self.logger.debug("VLC初始化成功")
            return True
        except Exception as e:
            self.logger.debug(f"VLC初始化失败: {e}")
            return False

    def _on_track_end(self, event):
        """曲目播放结束事件处理"""
        self.logger.debug("曲目播放结束事件")
        self.track_ended.emit()

    def play_file(self, file_url, file_name):
        """播放指定文件"""
        try:
            self.logger.debug(f"播放文件: {file_name}")
            # 创建 VLC 媒体（直接从 URL 流式播放）
            media = self.vlc_instance.media_new(file_url)
            # 默认启用音量标准化（固定为100%）
            media.add_option(":audio-filter=normvol")
            media.add_option(":norm-max-level=1.0")  # 固定为100%
            # 设置媒体并播放
            self.logger.debug("设置VLC媒体")
            self.vlc_player.set_media(media)
            self.logger.debug("开始播放音乐")
            self.vlc_player.play()
            # 更新状态
            self.playing = True
            self.paused = False
            self.current_track = file_name
            # 获取时长和更新UI
            self._get_vlc_duration()
            # 发送状态变化信号
            self.playback_state_changed.emit(self.playing, self.paused)
            self.logger.info(f"文件播放开始: {file_name}")
        except Exception as e:
            self.logger.debug(f"播放文件时出错: {e}")
            raise Exception(f"无法播放文件: {str(e)}")

    def _get_vlc_duration(self):
        """异步获取 VLC 媒体时长"""
        try:
            media = self.vlc_player.get_media()
            if media:
                duration_ms = media.get_duration()  # 直接获取毫秒
                if duration_ms > 0:
                    self.duration = duration_ms / 1000  # 转换为秒存储
                    # 发出时长更新信号（毫秒）
                    self.duration_changed.emit(duration_ms)
                    # 同时发出位置更新信号
                    self.position_changed.emit(int(self.position * 1000))
                else:
                    # 如果未获取到时长，500ms 后重试
                    from PyQt5.QtCore import QTimer
                    QTimer.singleShot(500, self._get_vlc_duration)
            else:
                self.logger.debug("无法获取媒体对象")
        except Exception as e:
            self.logger.debug(f"获取VLC媒体时长时出错: {e}")

    def toggle_play_pause(self):
        """切换播放/暂停状态"""
        call_id = int(time.time() * 1000)
        self.logger.info(
            f"[CALL_ID:{call_id}] MusicPlayer toggle_play_pause called, playing: {self.playing}, paused: {self.paused}")
        if not self.playing:
            self.logger.debug(f"[CALL_ID:{call_id}] No track playing, cannot toggle")
            return
        if self.playing:
            if self.paused:
                self.logger.debug(f"[CALL_ID:{call_id}] Resuming from pause")
                self.vlc_player.pause()
                self.paused = False
            else:
                self.logger.debug(f"[CALL_ID:{call_id}] Pausing playback")
                self.vlc_player.pause()
                self.paused = True
            self.logger.debug(f"[CALL_ID:{call_id}] Emitting state changed signal")
            self.playback_state_changed.emit(self.playing, self.paused)
        self.logger.debug(f"[CALL_ID:{call_id}] MusicPlayer toggle completed")

    def stop_playback(self):
        """停止播放"""
        self.logger.debug("停止播放")
        if self.playing:
            self.vlc_player.stop()
            self.playing = False
            self.paused = False
            self.position = 0
            # 发送状态变化信号
            self.playback_state_changed.emit(self.playing, self.paused)
            self.position_changed.emit(0)
            self.logger.debug("播放已停止")

    def set_position(self, position):
        """设置播放位置"""
        self.logger.debug(f"设置播放位置: {position}")
        if self.playing:
            self.vlc_player.set_time(int(position * 1000))  # 秒转毫秒
            self.position = position
            self.position_changed.emit(int(self.position))

    def set_volume(self, volume):
        """设置音量"""
        self.logger.debug(f"设置音量: {volume}")
        self.volume = volume
        self.vlc_player.audio_set_volume(volume)
        self.volume_changed.emit(volume)


    def get_current_time(self):
        """获取当前播放时间（毫秒）"""
        try:
            if self.playing and not self.paused:
                return self.vlc_player.get_time()
            return -1
        except:
            return -1

    def get_duration(self):
        """获取总时长（毫秒）"""
        try:
            if self.playing:
                media = self.vlc_player.get_media()
                if media:
                    return media.get_duration()
            return -1
        except:
            return -1

    def is_playing(self):
        """是否正在播放"""
        return self.playing

    def is_paused(self):
        """是否已暂停"""
        return self.paused

    def get_current_track(self):
        """获取当前播放曲目"""
        return self.current_track

    def release(self):
        """释放VLC资源"""
        try:
            if self.vlc_player:
                self.vlc_player.release()
                self.vlc_player = None
                self.logger.debug("VLC播放器已释放")
            if self.vlc_instance:
                self.vlc_instance.release()
                self.vlc_instance = None
                self.logger.debug("VLC实例已释放")
        except Exception as e:
            self.logger.debug(f"释放VLC资源时出错: {e}")

    def get_media_info(self):
        """获取媒体信息"""
        try:
            if self.playing:
                media = self.vlc_player.get_media()
                if media:
                    return {
                        'duration': media.get_duration(),
                        'mrl': media.get_mrl()
                    }
            return None
        except:
            return None

    def get_position(self):
        """获取当前播放位置（秒）"""
        try:
            if self.playing and not self.paused:
                return self.vlc_player.get_time() / 1000
            return -1
        except:
            return -1