# ===== 新增：日志管理模块 =====
import json
# ===== 新增：日志管理模块 =====
import os
import threading
from urllib.parse import quote

from PyQt5.QtCore import Qt, QUrl, pyqtSignal, QObject, QTimer
from PyQt5.QtGui import QColor
from PyQt5.QtWidgets import (QMessageBox, QInputDialog, QDialog,
                             QLineEdit, QListWidgetItem, QApplication)
from requests.auth import HTTPBasicAuth

from yc import PlaylistDialog,LogManager


class PlaylistManager(QObject):
    # 定义信号
    playlist_created = pyqtSignal(str)
    show_success_message = pyqtSignal(str)
    playlist_loaded_signal = pyqtSignal(dict)
    show_temp_status_message = pyqtSignal(str)
    # 新增UI更新信号
    update_playlist_list_signal = pyqtSignal()
    update_file_list_signal = pyqtSignal()
    enable_play_buttons_signal = pyqtSignal()
    set_status_label_signal = pyqtSignal(str)
    set_current_playlist_label_signal = pyqtSignal(str)
    set_current_index_signal = pyqtSignal(int)
    update_playlist_list_no_switch_signal = pyqtSignal()
    filter_results_ready = pyqtSignal(list, int)  # 过滤结果信号
    clear_search_box_signal = pyqtSignal()  # 新增：清空搜索框信号
    update_playlist_count_signal = pyqtSignal(int)  # 新增：歌单数量更新信号
    # 新增变量获取信号
    # get_current_playlist_signal = pyqtSignal()
    # get_playlist_list_signal = pyqtSignal()
    # get_allow_playlist_change_signal = pyqtSignal()
    # get_file_list_signal = pyqtSignal()


    def __init__(self, webdav_url, username, password, parent=None, app_dir=None):
        super().__init__(parent)
        self.logger = LogManager()
        self._webdav_url = webdav_url
        self._username = username
        self._password = password
        self.parent = parent  # 保存主窗口引用
        self.app_dir = app_dir or os.path.dirname(os.path.abspath(__file__))
        # 确保缓存目录存在
        cache_dir = os.path.join(self.app_dir, "cache")
        if not os.path.exists(cache_dir):
            os.makedirs(cache_dir)
        self._playlists = {"所有歌曲": [], "system": {}, "custom": {}}
        self._track_list = []
        self.track_dict = {}  # 新增：歌曲名到路径的映射字典
        self.track_list = []  # 添加对 track_list 的引用
        self._file_list = None  # 本地副本
        self._current_playlist = None  # 本地副本
        self._playlist_list = None  # 本地副本
        self.threads = []
        self.allow_playlist_change = True

        # 新增：自动创建歌单配置
        self.auto_create_playlists_from_folders = True
        self.load_playlist_settings()

        # 新增：初始化时加载track_dict
        self.load_track_dict_from_cache()

        # 初始化会话  每次访问的时候都会初始化，这里提前初始化没有意义
        # self.init_session()

    @property
    def webdav_url(self):
        """webdav_url 属性访问器"""
        if self.parent and hasattr(self.parent, 'webdav_url'):
            # 优先使用父实例的 webdav_url
            return self.parent.webdav_url
        return self._webdav_url

    @webdav_url.setter
    def webdav_url(self, value):
        """webdav_url 属性设置器"""
        # 同时更新父实例和本地副本
        if self.parent and hasattr(self.parent, 'webdav_url'):
            self.parent.webdav_url = value
        self._webdav_url = value

    @property
    def username(self):
        """username 属性访问器"""
        if self.parent and hasattr(self.parent, 'username'):
            # 优先使用父实例的 playlists
            return self.parent.username
        return self._username

    @username.setter
    def username(self, value):
        """playlists 属性设置器"""
        # 同时更新父实例和本地副本
        if self.parent and hasattr(self.parent, 'username'):
            self.parent.username = value
        self._username = value

    @property
    def password(self):
        """password 属性访问器"""
        if self.parent and hasattr(self.parent, 'password'):
            # 优先使用父实例的 playlists
            return self.parent.password
        return self._password

    @password.setter
    def password(self, value):
        """password 属性设置器"""
        # 同时更新父实例和本地副本
        if self.parent and hasattr(self.parent, 'password'):
            self.parent.password = value
        self._password = value

    @property
    def track_list(self):
        """track_list 属性访问器"""
        if self.parent and hasattr(self.parent, 'track_list'):
            # 优先使用父实例的 track_list
            return self.parent.track_list
        return self._track_list

    @track_list.setter
    def track_list(self, value):
        """track_list 属性设置器"""
        # 同时更新父实例和本地副本
        if self.parent and hasattr(self.parent, 'track_list'):
            self.parent.track_list = value
        self._track_list = value

    @property
    def playlists(self):
        """playlists 属性访问器"""
        if self.parent and hasattr(self.parent, 'playlists'):
            # 优先使用父实例的 playlists
            return self.parent.playlists
        return self._playlists

    @playlists.setter
    def playlists(self, value):
        """playlists 属性设置器"""
        # 同时更新父实例和本地副本
        if self.parent and hasattr(self.parent, 'playlists'):
            self.parent.playlists = value
        self._playlists = value

    @property
    def file_list(self):
        """file_list 属性访问器"""
        if self.parent and hasattr(self.parent, 'file_list'):
            # 关键：访问父实例的公共属性 file_list
            return self.parent.file_list
        return self._file_list

    @file_list.setter
    def file_list(self, value):
        """file_list 属性设置器"""
        # 关键：设置父实例的公共属性 file_list
        if self.parent and hasattr(self.parent, 'file_list'):
            self.parent.file_list = value
        self._file_list = value

    @property
    def current_playlist(self):
        """current_playlist 属性访问器"""
        if self.parent and hasattr(self.parent, 'current_playlist'):
            # 关键：访问父实例的公共属性 current_playlist
            return self.parent.current_playlist
        return self._current_playlist

    @current_playlist.setter
    def current_playlist(self, value):
        """current_playlist 属性设置器"""
        # 关键：设置父实例的公共属性 current_playlist
        if self.parent and hasattr(self.parent, 'current_playlist'):
            self.parent.current_playlist = value
        self._current_playlist = value

    @property
    def playlist_list(self):
        """playlist_list 属性访问器"""
        if self.parent and hasattr(self.parent, 'playlist_list'):
            # 关键：访问父实例的公共属性 playlist_list
            return self.parent.playlist_list
        return self._playlist_list

    @playlist_list.setter
    def playlist_list(self, value):
        """playlist_list 属性设置器"""
        # 关键：设置父实例的公共属性 playlist_list
        if self.parent and hasattr(self.parent, 'playlist_list'):
            self.parent.playlist_list = value
        self._playlist_list = value

    def load_track_dict_from_cache(self):
        """从缓存加载track_dict"""
        try:
            cache_dir = os.path.join(self.app_dir, "cache")
            cache_file = os.path.join(cache_dir, f"{self.username}_track_list.json")
            if os.path.exists(cache_file):
                with open(cache_file, 'r', encoding='utf-8') as f:
                    track_list = json.load(f)
                self.track_dict = {track['name']: track for track in track_list}
                self.track_list = track_list  # 同时更新track_list
                self.logger.debug(f"从缓存加载track_dict成功，共 {len(self.track_dict)} 首歌曲")
                # 同步到父窗口
                if self.parent:
                    self.parent.track_dict = self.track_dict
                    self.parent.track_list = self.track_list
                return True
            else:
                self.logger.debug("track_list缓存文件不存在")
                return False
        except Exception as e:
            self.logger.debug(f"加载track_dict失败: {e}")
            return False


    def build_webdav_url(self, path="", include_auth=False):
        """构建WebDAV URL的公共方法"""
        # 使用QUrl来正确解析和构建URL
        parsed_url = QUrl(self.webdav_url)
        scheme = parsed_url.scheme()
        host = parsed_url.host()
        port = parsed_url.port()
        if port == -1:
            port = 80 if scheme == 'http' else 443
        # 获取WebDAV配置中的路径
        webdav_path = parsed_url.path()
        # 确保路径以/开头但不以/结尾（除了根目录）
        if webdav_path and not webdav_path.startswith('/'):
            webdav_path = '/' + webdav_path
        if webdav_path and webdav_path.endswith('/'):
            webdav_path = webdav_path[:-1]
        # 确保path以/开头
        if path and not path.startswith('/'):
            path = '/' + path
        # 构建基础URL
        base_url = f"{scheme}://{host}:{port}{webdav_path}{path}"
        # 如果需要包含认证信息
        if include_auth:
            encoded_username = quote(self.username)
            encoded_password = quote(self.password)
            base_url = f"{scheme}://{encoded_username}:{encoded_password}@{host}:{port}{webdav_path}{path}"
        return base_url

    def save_playlists(self):
        """保存歌单到WebDAV服务器"""
        self.logger.info("保存歌单到WebDAV服务器")
        try:
            playlist_filename = f"{self.username}_playlists.json"
            upload_url = self.build_webdav_url(f"/{playlist_filename}", include_auth=False)
            self.logger.debug(f"歌单上传URL: {upload_url}")

            session = self.get_session()
            if session is None:
                self.logger.debug("无法获取HTTP会话")
                return False

            headers = {'Content-Type': 'application/json'}
            response = session.put(
                upload_url,
                data=json.dumps(self.playlists, ensure_ascii=False).encode('utf-8'),
                headers=headers,
                timeout=10
            )

            if response.status_code in (200, 201, 204):
                self.logger.debug(f"歌单上传成功: {playlist_filename}")
                # 新增：验证服务器保存结果
                verify_response = session.get(upload_url, timeout=10)
                if verify_response.status_code == 200:
                    server_data = json.loads(verify_response.content.decode('utf-8'))
                    if server_data == self.playlists:
                        self.logger.debug("服务器数据验证成功")
                        self.save_playlists_local_cache()  # 同步保存本地缓存
                        return True
                    else:
                        self.logger.warning("服务器数据验证失败，数据不一致")
                        return False
                else:
                    self.logger.debug("无法验证服务器数据")
                    return False
            else:
                self.logger.debug(f"歌单上传失败，状态码: {response.status_code}")
                return False
        except Exception as e:
            self.logger.debug(f"保存歌单失败: {e}")
            return False

    def save_playlists_local_cache(self):
        """保存歌单到本地缓存"""
        cache_dir = os.path.join(self.app_dir, "cache")
        if not os.path.exists(cache_dir):
            os.makedirs(cache_dir)
        cache_file = os.path.join(cache_dir, f"{self.username}_playlists.json")
        try:
            with open(cache_file, 'w', encoding='utf-8') as f:
                json.dump(self.playlists, f, ensure_ascii=False, indent=2)
            self.logger.debug(f"歌单已保存到本地缓存: {cache_file}")

            # 新增：验证写入结果
            with open(cache_file, 'r', encoding='utf-8') as f:
                saved_data = json.load(f)
                if saved_data != self.playlists:
                    self.logger.warning("本地缓存验证失败，数据不一致")
                    return False

            self.logger.debug("本地缓存验证成功")
            return True
        except Exception as e:
            self.logger.debug(f"保存本地缓存失败: {e}")
            return False

    def load_playlists(self):
        """从WebDAV服务器加载歌单"""
        self.logger.debug("从WebDAV服务器加载歌单")
        try:
            session = self.get_session()
            if session is None:
                self._load_from_local_cache_or_default()
                return

            playlist_filename = f"{self.username}_playlists.json"
            download_url = self.build_webdav_url(f"/{playlist_filename}", include_auth=False)
            self.logger.debug(f"歌单下载URL: {download_url}")

            try:
                response = session.get(download_url, timeout=10)
                if response.status_code == 200:
                    server_playlists = json.loads(response.content.decode('utf-8'))
                    self.logger.debug(f"歌单加载成功，共 {len(server_playlists)} 个歌单")

                    # 关键修复：确保结构完整且验证数据
                    if "所有歌曲" not in server_playlists:
                        server_playlists["所有歌曲"] = []
                    if "system" not in server_playlists:
                        server_playlists["system"] = {}
                    if "custom" not in server_playlists:
                        server_playlists["custom"] = {}

                    # 验证系统歌单数据有效性
                    for playlist_name, songs in server_playlists["system"].items():
                        if not isinstance(songs, list):
                            self.logger.warning(f"系统歌单 {playlist_name} 数据格式错误，重置为空列表")
                            server_playlists["system"][playlist_name] = []

                    self.playlists = server_playlists
                    self.save_playlists_local_cache()

                    # 关键修复：确保track_list已加载
                    if not self.track_list:
                        self.load_track_list_from_local_cache()

                else:
                    self.logger.debug(f"歌单文件不存在，状态码: {response.status_code}")
                    self._load_from_local_cache_or_default()
                    if response.status_code == 404:
                        self.save_playlists()
            except Exception as e:
                self.logger.debug(f"从WebDAV加载歌单时出错: {e}")
                self._load_from_local_cache_or_default()
        except Exception as e:
            self.logger.debug(f"加载歌单失败: {e}")
            import traceback
            traceback.print_exc()
            self._load_from_local_cache_or_default()


    def _load_from_local_cache_or_default(self):
        """从本地缓存加载歌单或创建默认歌单"""
        if self.load_playlists_from_local_cache():
            self.logger.debug("从本地缓存加载歌单成功")
        else:
            # 创建默认歌单
            self.playlists = {"所有歌曲": [], "system": {}, "custom": {}}
            self.logger.debug("创建默认歌单")

    def load_track_list_from_local_cache(self):
        """从本地缓存加载音乐文件列表"""
        self.logger.debug("尝试从本地缓存加载音乐文件列表")
        try:
            cache_dir = os.path.join(self.app_dir, "cache")
            cache_file = os.path.join(cache_dir, f"{self.username}_track_list.json")
            if os.path.exists(cache_file):
                with open(cache_file, 'r', encoding='utf-8') as f:
                    self.track_list = json.load(f)
                # 新增：构建歌曲名到路径的映射字典
                self.track_dict = {track['name']: track for track in self.track_list}
                # 同步到父窗口
                if self.parent:
                    self.parent.track_dict = self.track_dict
                    self.parent.track_list = self.track_list

                # 检查音乐文件列表是否为空
                if not self.track_list or len(self.track_list) == 0:
                    self.logger.debug("本地缓存音乐文件列表为空，视为无效缓存")
                    return False
                # 更新所有歌曲歌单
                self.playlists["所有歌曲"] = [track['name'] for track in self.track_list]
                self.logger.debug(f"从本地缓存加载音乐文件列表成功，共 {len(self.track_list)} 个文件")
                return True
            else:
                self.logger.debug("本地音乐文件列表缓存文件不存在")
                return False
        except Exception as e:
            self.logger.debug(f"从本地缓存加载音乐文件列表失败: {e}")
            return False

    def filter_songs(self, keyword, current_playlist, playing_track=None):
        """根据关键字过滤歌曲列表"""
        # 如果关键字为空，则显示当前歌单的所有歌曲
        if not keyword.strip():
            self.update_file_list_signal.emit()
            return

        # 转换为小写，不区分大小写
        keyword = keyword.lower()

        # 获取当前歌单的歌曲列表
        songs_list = []
        if current_playlist == "所有歌曲":
            songs_list = self.playlists.get("所有歌曲", [])
        elif current_playlist in self.playlists.get("system", {}):
            songs_list = self.playlists["system"][current_playlist]
        elif current_playlist in self.playlists.get("custom", {}):
            songs_list = self.playlists["custom"][current_playlist]

        # 准备过滤结果
        filtered_results = []
        added_songs = set()
        current_song_index = -1

        # 遍历当前歌单的所有歌曲
        for song_name in songs_list:
            # 如果歌曲名包含关键字（不区分大小写）
            if keyword in song_name.lower():
                # 检查是否已经添加过，避免重复
                if song_name in added_songs:
                    continue
                # 关键优化：使用字典进行O(1)查找
                if song_name in self.track_dict:
                    track = self.track_dict[song_name]
                    filtered_results.append({
                        'name': song_name,
                        'path': track['path'],
                        'display_text': f"{len(filtered_results) + 1}. {song_name}"
                    })
                    added_songs.add(song_name)
                    # 检查是否是当前播放的歌曲
                    if playing_track and playing_track == song_name:
                        current_song_index = len(filtered_results) - 1

        # 发送过滤结果到主窗口
        self.filter_results_ready.emit(filtered_results, current_song_index)


    def load_playlists_async(self):
        """异步加载歌单 - 从WebDAVMusicPlayer迁移过来的方法"""
        self.logger.debug("开始异步加载歌单")

        # 使用后台线程加载歌单
        load_thread = threading.Thread(target=self.load_playlists_with_priority, daemon=True)
        load_thread.start()

    def load_playlists_with_priority(self):
        """内部方法：优先从本地加载歌单，同时显示本地缓存 - 优化版"""
        self.logger.debug("优先从本地加载歌单，同时显示本地缓存")
        try:
            # 首先尝试从本地缓存加载歌单，以便立即显示历史数据
            local_playlists_loaded = self.load_playlists_from_local_cache()
            # 尝试从本地缓存加载音乐文件列表
            local_track_list_loaded = self.load_track_list_from_local_cache()
            # 检查缓存是否有效（非空）
            cache_valid = local_playlists_loaded and local_track_list_loaded

            if cache_valid:
                self.logger.debug("本地歌单缓存加载成功，立即显示UI")
                # 立即更新UI显示本地缓存
                self.set_status_label_signal.emit(f"已连接: {self.webdav_url} (正在从本地缓存中更新歌单...)")

                # 优化1：优先显示歌单列表（用户可以立即操作）
                self.update_playlist_list_signal.emit()
                self.logger.debug("歌单列表已更新")

                # 优化2：延迟更新文件列表，让歌单列表先显示
                QTimer.singleShot(50, self._delayed_update_file_list)

                # 优化3：启用播放控制按钮（用户可以开始播放）
                self.enable_play_buttons_signal.emit()
                self.logger.debug("播放控制按钮已启用")

                # 优化4：延迟恢复状态标签，避免过早覆盖
                QTimer.singleShot(2000, lambda: self.set_status_label_signal.emit(f"已连接: {self.webdav_url}"))

                return cache_valid  # 本地缓存加载后不再扫描服务器歌单
            else:
                self.logger.debug("本地歌单缓存不存在或加载失败")
                # 使用默认歌单
                self.playlists = {"所有歌曲": [], "system": {}, "custom": {}}

                # 优化5：即使没有缓存，也先显示空列表
                self.update_playlist_list_signal.emit()
                self.update_file_list_signal.emit()
                self.set_status_label_signal.emit(f"已连接: {self.webdav_url} (正在从服务器加载歌单...)")
                QTimer.singleShot(3000, lambda: self.set_status_label_signal.emit(f"已连接: {self.webdav_url}"))

            # 然后在后台线程中从服务器加载最新歌单
            def load_server_playlists():
                try:
                    # 确保会话已初始化
                    session = self.get_session()
                    # 构建歌单文件路径（基于用户名）
                    playlist_filename = f"{self.username}_playlists.json"
                    # 解析WebDAV URL，获取协议、主机、端口和路径
                    parsed_url = QUrl(self.webdav_url)
                    scheme = parsed_url.scheme()
                    host = parsed_url.host()
                    port = parsed_url.port()
                    if port == -1:
                        port = 80 if scheme == 'http' else 443
                    # 获取WebDAV配置中的路径
                    webdav_path = parsed_url.path()
                    # 确保路径以/开头但不以/结尾（除了根目录）
                    if webdav_path and not webdav_path.startswith('/'):
                        webdav_path = '/' + webdav_path
                    if webdav_path and webdav_path.endswith('/'):
                        webdav_path = webdav_path[:-1]
                    # 对文件名进行URL编码
                    encoded_playlist_filename = quote(playlist_filename)
                    # 构建完整的下载URL（包含WebDAV配置中的路径）
                    encoded_username = quote(self.username)
                    encoded_password = quote(self.password)
                    download_url = f"{scheme}://{encoded_username}:{encoded_password}@{host}:{port}{webdav_path}/{encoded_playlist_filename}"
                    self.logger.debug(f"歌单下载URL: {download_url}")
                    try:
                        response = session.get(download_url, timeout=10)
                        if response.status_code == 200:
                            server_playlists = json.loads(response.content.decode('utf-8'))
                            self.logger.debug(f"服务器歌单加载成功，共 {len(server_playlists)} 个歌单")
                            # 确保所有歌曲歌单存在
                            if "所有歌曲" not in server_playlists:
                                server_playlists["所有歌曲"] = []
                            # 优化6：发送信号，在主线程中更新UI（延迟执行避免阻塞）
                            QTimer.singleShot(100, lambda: self.playlist_loaded_signal.emit(server_playlists))
                        else:
                            self.logger.debug(f"服务器歌单文件不存在，状态码: {response.status_code}")
                            # 使用本地缓存的歌单
                            QTimer.singleShot(100, lambda: self.playlist_loaded_signal.emit(self.playlists))
                            # 服务器缓存不存在的话，就需要刷新了，重新扫描文件
                            # 注意：这里需要通过信号调用main1.py中的connect_and_load方法
                    except Exception as e:
                        self.logger.debug(f"从服务器加载歌单时出错: {e}")
                        # 使用本地缓存的歌单
                        QTimer.singleShot(100, lambda: self.playlist_loaded_signal.emit(self.playlists))
                except Exception as e:
                    self.logger.debug(f"后台加载服务器歌单失败: {e}")
                    # 使用本地缓存的歌单
                    QTimer.singleShot(100, lambda: self.playlist_loaded_signal.emit(self.playlists))

            # 启动后台线程加载服务器歌单（仅当本地缓存无效时）
            if not cache_valid:
                self.logger.info("本地无有效缓存，扫描服务器歌单")
                server_load_thread = threading.Thread(target=load_server_playlists, daemon=True)
                server_load_thread.start()
                self.threads.append(server_load_thread)
                return cache_valid
        except Exception as e:
            self.logger.debug(f"优先加载歌单失败: {e}")
            import traceback
            traceback.print_exc()
            # 使用默认歌单
            self.playlists = {"所有歌曲": []}
            self.update_playlist_list_signal.emit()
            self.update_file_list_signal.emit()

    def _delayed_update_file_list(self):
        """延迟更新文件列表的辅助方法"""
        try:
            self.logger.debug("开始延迟更新文件列表")
            # 更新状态标签提示用户
            self.set_status_label_signal.emit(f"已连接: {self.webdav_url} (正在加载歌曲列表...)")

            # 发送更新文件列表信号
            self.update_file_list_signal.emit()

            # 延迟恢复最终状态
            QTimer.singleShot(1500, lambda: self.set_status_label_signal.emit(f"已连接: {self.webdav_url}"))

        except Exception as e:
            self.logger.debug(f"延迟更新文件列表失败: {e}")

    def load_playlists_from_local_cache(self):
        """从本地缓存加载歌单"""
        self.logger.debug("尝试从本地缓存加载歌单")
        try:
            cache_dir = os.path.join(self.app_dir, "cache")
            cache_file = os.path.join(cache_dir, f"{self.username}_playlists.json")
            if os.path.exists(cache_file):
                with open(cache_file, 'r', encoding='utf-8') as f:
                    self.playlists = json.load(f)
                # 检查所有歌曲歌单是否为空
                if not self._playlists["所有歌曲"] or len(self._playlists["所有歌曲"]) == 0:
                    self.logger.debug("本地缓存中所有歌曲歌单为空，视为无效缓存")
                    return False
                self.logger.debug(f"从本地缓存加载歌单成功，共 {len(self.playlists)} 个歌单")
                return True
            else:
                self.logger.debug("本地缓存文件不存在")
                return False
        except Exception as e:
            self.logger.debug(f"从本地缓存加载歌单失败: {e}")
            return False


    def save_track_list_local_cache(self):
        """保存音乐文件列表到本地缓存"""
        cache_dir = os.path.join(self.app_dir, "cache")
        if not os.path.exists(cache_dir):
            os.makedirs(cache_dir)
        cache_file = os.path.join(cache_dir, f"{self.username}_track_list.json")
        try:
            with open(cache_file, 'w', encoding='utf-8') as f:
                json.dump(self.track_list, f, ensure_ascii=False, indent=2)
            self.logger.debug(f"音乐文件列表已保存到本地缓存: {cache_file}")
        except Exception as e:
            self.logger.debug(f"保存音乐文件列表本地缓存失败: {e}")


    def create_playlist(self):
        """创建新歌单"""
        self.logger.info("创建新歌单")
        try:
            # 检查是否有父窗口引用
            if self.parent is None:
                self.logger.debug("没有父窗口引用，无法创建对话框")
                return
            # 获取现有歌单名称列表（不包括"所有歌曲"和系统歌单）
            existing_playlists = list(self.playlists["custom"].keys())

            dialog = PlaylistDialog(self.parent, existing_playlists)
            self.logger.debug("显示创建歌单对话框")
            if dialog.exec_() == QDialog.Accepted:
                playlist_name = dialog.get_playlist_name()
                self.logger.info(f"用户确认创建歌单: {playlist_name}")
                self.playlists["custom"][playlist_name] = []
                # 保存到WebDAV服务器
                if self.save_playlists():
                    self.logger.debug("歌单已保存到服务器")
                    # 发送信号，而不是直接调用方法
                    self.playlist_created.emit(playlist_name)
                else:
                    QMessageBox.warning(self.parent, "警告", "歌单创建成功，但保存到服务器失败")
            else:
                self.logger.debug("用户取消创建歌单")
        except Exception as e:
            self.logger.debug(f"创建歌单时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.parent, "错误", f"创建歌单失败: {str(e)}")

    def delete_playlist(self):
        """删除当前歌单"""
        self.logger.info("删除当前歌单")
        try:
            selected_items = self.parent.playlist_list.selectedItems()
            if not selected_items:
                self.logger.debug("没有选中的歌单")
                return
            if len(selected_items) > 1:
                selected_playlist_names = [item.text() for item in selected_items]
                deletable_playlists = []
                for name in selected_playlist_names:
                    if name != "所有歌曲":
                        if name in self.playlists.get("system", {}) or name in self.playlists.get("custom", {}):
                            deletable_playlists.append(name)
                if deletable_playlists:
                    self.delete_selected_playlists(deletable_playlists)
                return
            playlist_name = selected_items[0].text()
            if playlist_name == "所有歌曲":
                QMessageBox.warning(self.parent, "警告", "不能删除所有歌曲歌单")
                return
            # 新增：检查歌单是否存在于子字典中
            playlist_found = False
            if playlist_name in self.playlists.get("system", {}):
                del self.playlists["system"][playlist_name]
                playlist_found = True
            elif playlist_name in self.playlists.get("custom", {}):
                del self.playlists["custom"][playlist_name]
                playlist_found = True
            if not playlist_found:
                self.logger.debug(f"歌单不存在: {playlist_name}")
                return
            reply = QMessageBox.question(
                self.parent, "确认删除",
                f"确定要删除歌单 '{playlist_name}' 吗?",
                QMessageBox.Yes | QMessageBox.No
            )
            if reply == QMessageBox.Yes:
                self.logger.info(f"用户确认删除歌单: {playlist_name}")
                if self.save_playlists():
                    self.set_status_label_signal.emit(f"歌单 '{playlist_name}' 删除成功")
                    self.logger.debug(f"歌单删除并保存到服务器成功")
                    if self.current_playlist == playlist_name:
                        self.set_current_playlist_label_signal.emit("所有歌曲")
                    self.update_playlist_list_signal.emit()
                    self.update_file_list_signal.emit()
                    self.set_current_index_signal.emit(-1)
                    self.show_temp_status_message.emit(f"歌单 '{playlist_name}' 删除成功")
                else:
                    QMessageBox.warning(self.parent, "警告", "歌单删除成功，但保存到服务器失败")
            else:
                self.logger.debug("用户取消删除歌单")
        except Exception as e:
            self.logger.debug(f"删除歌单时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.parent, "错误", f"删除歌单失败: {str(e)}")

    def delete_selected_playlists(self, playlist_names):
        """删除多个选中的歌单"""
        self.logger.info(f"删除多个选中的歌单: {playlist_names}")
        try:
            if not playlist_names:
                self.logger.debug("没有要删除的歌单")
                return
            valid_playlists = []
            for name in playlist_names:
                if name == "所有歌曲":
                    self.logger.debug("不能删除所有歌曲歌单")
                    continue
                # 新增：检查子字典中的歌单
                if name in self.playlists.get("system", {}) or name in self.playlists.get("custom", {}):
                    valid_playlists.append(name)
                else:
                    self.logger.debug(f"歌单不存在: {name}")
            if not valid_playlists:
                self.logger.debug("没有有效的歌单可删除")
                return
            if len(valid_playlists) == 1:
                message = f"确定要删除歌单 '{valid_playlists[0]}' 吗?"
            else:
                playlist_list = "\n".join([f"• {name}" for name in valid_playlists])
                message = f"确定要删除以下 {len(valid_playlists)} 个歌单吗?\n\n{playlist_list}"
            reply = QMessageBox.question(
                self.parent, "确认删除",
                message,
                QMessageBox.Yes | QMessageBox.No
            )
            if reply == QMessageBox.Yes:
                self.logger.info(f"用户确认删除 {len(valid_playlists)} 个歌单")
                deleted_count = 0
                for playlist_name in valid_playlists:
                    # 新增：从对应子字典中删除
                    if playlist_name in self.playlists.get("system", {}):
                        del self.playlists["system"][playlist_name]
                        deleted_count += 1
                        self.logger.debug(f"已删除系统歌单: {playlist_name}")
                    elif playlist_name in self.playlists.get("custom", {}):
                        del self.playlists["custom"][playlist_name]
                        deleted_count += 1
                        self.logger.debug(f"已删除自定义歌单: {playlist_name}")
                if self.save_playlists():
                    self.set_status_label_signal.emit(f"已成功删除 {deleted_count} 个歌单")
                    self.logger.debug(f"歌单删除并保存到服务器成功，共删除 {deleted_count} 个")
                    if self.current_playlist in valid_playlists:
                        self.set_current_playlist_label_signal.emit("所有歌曲")
                    self.update_playlist_list_signal.emit()
                    self.update_file_list_signal.emit()
                    self.set_current_index_signal.emit(-1)
                    self.show_temp_status_message.emit(f"已成功删除 {deleted_count} 个歌单")
                else:
                    QMessageBox.warning(self.parent, "警告", "歌单删除成功，但保存到服务器失败")
            else:
                self.logger.debug("用户取消删除歌单")
        except Exception as e:
            self.logger.debug(f"删除多个歌单时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.parent, "错误", f"删除歌单失败: {str(e)}")

    def rename_playlist(self):
        """重命名当前歌单"""
        self.logger.info("重命名当前歌单")
        try:
            if self.current_playlist == "所有歌曲":
                self.logger.debug("不能重命名所有歌曲歌单")
                QMessageBox.warning(self, "警告", "不能重命名'所有歌曲'歌单")
                return

            current_item = self.playlist_list.currentItem()
            if not current_item:
                self.logger.debug("没有选中的歌单")
                return

            old_name = current_item.text()
            self.logger.debug(f"当前选中的歌单: {old_name}")

            new_name, ok = QInputDialog.getText(
                self.parent, "重命名歌单",
                f"请输入新的歌单名称:",
                QLineEdit.Normal,
                old_name
            )

            if not ok or not new_name:
                self.logger.debug("用户取消重命名")
                return

            new_name = new_name.strip()
            if not new_name:
                QMessageBox.warning(self.parent, "警告", "歌单名称不能为空")
                return

            if new_name == old_name:
                self.logger.debug("歌单名称未改变")
                return

            # 检查新名称是否已存在（遍历所有子字典）
            if (new_name in self.playlists.get("system", {}) or
                    new_name in self.playlists.get("custom", {})):
                QMessageBox.warning(self.parent, "警告", f"歌单名称 '{new_name}' 已存在")
                return

            # 关键修复：从子字典中查找并移动歌单
            playlist_found = False
            source_dict = None

            if old_name in self.playlists.get("system", {}):
                source_dict = self.playlists["system"]
                playlist_found = True
            elif old_name in self.playlists.get("custom", {}):
                source_dict = self.playlists["custom"]
                playlist_found = True

            if not playlist_found:
                self.logger.debug(f"歌单不存在: {old_name}")
                QMessageBox.warning(self.parent, "错误", f"歌单 '{old_name}' 不存在")
                return

            # 执行重命名操作
            source_dict[new_name] = source_dict[old_name]
            del source_dict[old_name]

            # 更新当前播放的歌单名称
            if self.current_playlist == old_name:
                self.set_current_playlist_label_signal.emit(new_name)

            if self.save_playlists():
                self.logger.debug("歌单重命名并保存到服务器成功")
                self.update_playlist_list_no_switch_signal.emit()
                self.update_file_list_signal.emit()
                self.logger.info(f"歌单重命名完成: {old_name} -> {new_name}")
                self.show_temp_status_message.emit(f"歌单已重命名为: {new_name}")
            else:
                QMessageBox.warning(self.parent, "警告", "歌单重命名成功，但保存到服务器失败")
        except Exception as e:
            self.logger.debug(f"重命名歌单时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.parent, "错误", f"重命名歌单失败: {str(e)}")

    def reset_playlists(self):
        """重置歌单功能"""
        self.logger.info("开始重置歌单")
        try:
            if self.parent is None:
                self.logger.debug("没有父窗口引用，无法创建对话框")
                return

            reply = QMessageBox.question(
                self.parent,
                "重置歌单",
                "是否采用文件夹结构自动创建歌单？\n\n"
                "选择“是”将根据文件夹结构创建歌单（需要重新扫描服务器）\n"
                "选择“否”将清空所有歌单，只保留“所有歌曲”",
                QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel
            )
            if reply == QMessageBox.Cancel:
                self.logger.debug("用户取消重置歌单")
                return

            # 关键修复1：备份并修改数据
            custom_playlists_backup = self.playlists.get("custom", {}).copy()
            all_songs_backup = self.playlists.get("所有歌曲", []).copy()

            if reply == QMessageBox.No:
                # 清空所有歌单（保留所有歌曲）
                self.playlists = {
                    "所有歌曲": all_songs_backup,
                    "system": {},  # 清空系统歌单
                    "custom": custom_playlists_backup  # 保留用户歌单
                }
                self.logger.info("用户选择清空歌单")
                self.show_temp_status_message.emit("歌单已重置，只保留所有歌曲")

                # 关键修复2：立即保存到本地缓存和服务器
                self.save_playlists_local_cache()  # 保存本地缓存
                if self.save_playlists():  # 保存到WebDAV
                    self.logger.debug("重置后的歌单已保存到服务器")
                    self.update_playlist_list_signal.emit()
                    self.update_file_list_signal.emit()
                    self.set_current_index_signal.emit(-1)
                    if self.current_playlist != "所有歌曲":
                        self.set_current_playlist_label_signal.emit("所有歌曲")
                else:
                    QMessageBox.warning(self.parent, "警告", "歌单重置成功，但保存到服务器失败")

            elif reply == QMessageBox.Yes:
                # 清空系统歌单，保留自定义歌单
                self.playlists["system"] = {}
                self.logger.info("用户选择采用文件夹结构创建歌单")

                # 关键修复3：立即保存清空后的状态
                self.save_playlists_local_cache()
                if self.save_playlists():
                    self.logger.debug("系统歌单已清空并保存到服务器")
                else:
                    self.logger.debug("保存清空状态失败")

                if hasattr(self.parent, 'connect_and_load'):
                    self.set_status_label_signal.emit("正在重新扫描服务器上的音乐文件...")
                    self.parent.connect_and_load()
                    self._delayed_create_playlists = QTimer(self)
                    self._delayed_create_playlists.timeout.connect(self._create_playlists_after_refresh)
                    self._delayed_create_playlists.setSingleShot(True)
                    self._delayed_create_playlists.start(5000)
                else:
                    self._create_playlists_after_refresh()

        except Exception as e:
            self.logger.debug(f"重置歌单时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.parent, "错误", f"重置歌单失败: {str(e)}")

    def _create_playlists_after_refresh(self):
        """刷新音乐文件后创建歌单"""
        try:
            created_count = self.create_playlists_from_folders(
                self.track_list,
                enabled=True,
                force_create=True
            )
            if created_count > 0:
                self.show_temp_status_message.emit(f"已根据文件夹结构自动创建 {created_count} 个歌单")
            else:
                self.show_temp_status_message.emit("未找到合适的文件夹结构创建歌单")

            # 关键修复4：创建完成后立即保存
            self.save_playlists_local_cache()
            if self.save_playlists():
                self.logger.debug("新建的系统歌单已保存到服务器")
                self.update_playlist_list_signal.emit()
                self.update_file_list_signal.emit()
            else:
                QMessageBox.warning(self.parent, "警告", "系统歌单创建成功，但保存到服务器失败")

        except Exception as e:
            self.logger.debug(f"刷新后创建歌单时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.parent, "错误", f"重置歌单失败: {str(e)}")

    def _save_reset_playlists(self):
        """保存重置后的歌单"""
        self.save_playlists_local_cache()
        if self.save_playlists():
            self.logger.debug("歌单已保存到服务器")
            # 更新UI
            self.update_playlist_list_signal.emit()
            self.update_file_list_signal.emit()
            # 重置当前播放索引
            self.set_current_index_signal.emit(-1)
            # 如果当前歌单不是所有歌曲，切换到所有歌曲
            if self.current_playlist != "所有歌曲":
                self.set_current_playlist_label_signal.emit("所有歌曲")
        else:
            QMessageBox.warning(self.parent, "警告", "歌单重置成功，但保存到服务器失败")

    def add_to_playlist(self, playlist_name):
        """将选中的歌曲添加到指定歌单"""
        self.logger.debug(f"将歌曲添加到歌单 '{playlist_name}'")
        try:
            selected_items = self.file_list.selectedItems()
            if not selected_items:
                self.logger.debug("没有选中的歌曲")
                return

            # 关键修复：检查歌单类型及权限
            target_playlist = None
            playlist_type = None

            if playlist_name == "所有歌曲":
                self.logger.debug("不能向'所有歌曲'歌单添加歌曲")
                QMessageBox.warning(self.parent, "提示", "不能向'所有歌曲'歌单添加歌曲")
                return
            elif playlist_name in self.playlists.get("system", {}):
                # 系统歌单：不允许添加歌曲
                self.logger.warning(f"系统歌单 '{playlist_name}' 不允许添加歌曲")
                QMessageBox.warning(self.parent, "提示", f"系统歌单 '{playlist_name}' 不允许添加歌曲")
                return
            elif playlist_name in self.playlists.get("custom", {}):
                target_playlist = self.playlists["custom"][playlist_name]
                playlist_type = "custom"

            # 验证歌单是否存在
            if target_playlist is None:
                self.logger.warning(f"歌单 '{playlist_name}' 不存在或不可操作")
                QMessageBox.warning(self.parent, "错误", f"歌单 '{playlist_name}' 不存在")
                return

            added_count = 0
            already_exists_count = 0

            for item in selected_items:
                song_name = item.data(Qt.UserRole + 1)
                if song_name not in target_playlist:
                    target_playlist.append(song_name)
                    added_count += 1
                    self.logger.debug(f"添加歌曲到自定义歌单: {song_name}")
                else:
                    already_exists_count += 1
                    self.logger.debug(f"歌曲已存在于自定义歌单: {song_name}")

            # 保存到WebDAV服务器
            if self.save_playlists():
                self.logger.debug("歌单已保存到服务器")
                # 根据操作结果显示不同的消息
                if added_count > 0 and already_exists_count > 0:
                    message = f"已添加 {added_count} 首歌曲到 '{playlist_name}'，{already_exists_count} 首已存在"
                elif added_count > 0:
                    message = f"已添加 {added_count} 首歌曲到 '{playlist_name}'"
                elif already_exists_count > 0:
                    message = f"选中的歌曲已在 '{playlist_name}' 中"
                else:
                    message = "没有歌曲被添加"

                self.show_temp_status_message.emit(message)
                # 更新当前文件列表（如果正在查看该歌单）
                if self.current_playlist == playlist_name:
                    self.update_file_list_signal.emit()
            else:
                QMessageBox.warning(self.parent, "警告", "歌曲添加成功，但保存到服务器失败")

        except Exception as e:
            self.logger.debug(f"添加歌曲到歌单时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.parent, "错误", f"添加歌曲失败: {str(e)}")

    def remove_from_playlist(self):
        """从当前歌单移除选中的歌曲"""
        self.logger.debug("从当前歌单移除选中的歌曲")
        try:
            selected_items = self.file_list.selectedItems()
            if not selected_items:
                return

            current_playlist = self.current_playlist
            self.logger.debug(f"从歌单 '{current_playlist}' 移除 {len(selected_items)} 首歌曲")

            # 关键修复：根据歌单类型确定访问路径
            target_playlist = None

            if current_playlist == "所有歌曲":
                self.logger.debug("不能从'所有歌曲'歌单移除歌曲")
                QMessageBox.warning(self.parent, "提示", "不能从'所有歌曲'歌单移除歌曲")
                return
            elif current_playlist in self.playlists.get("system", {}):
                # 系统歌单：允许删除歌曲
                target_playlist = self.playlists["system"][current_playlist]
            elif current_playlist in self.playlists.get("custom", {}):
                target_playlist = self.playlists["custom"][current_playlist]

            if target_playlist is None:
                self.logger.warning(f"当前歌单 '{current_playlist}' 不存在")
                return

            removed_count = 0
            for item in selected_items:
                song_name = item.data(Qt.UserRole + 1)
                if song_name in target_playlist:
                    target_playlist.remove(song_name)
                    removed_count += 1
                    self.logger.debug(f"从歌单移除歌曲: {song_name}")

            # 保存到WebDAV服务器
            if self.save_playlists():
                self.logger.debug("歌单已保存到服务器")
                if removed_count > 0:
                    self.update_file_list_signal.emit()
                    message = f"已从 '{current_playlist}' 移除 {removed_count} 首歌曲"
                else:
                    message = "没有歌曲被移除"
                self.show_temp_status_message.emit(message)
            else:
                QMessageBox.warning(self, "警告", "歌曲移除成功，但保存到服务器失败")
        except Exception as e:
            self.logger.debug(f"从歌单移除歌曲时出错: {e}")
            import traceback
            traceback.print_exc()


    def cleanup_local_cache(self):
        """清理本地缓存（删除非当前用户的缓存文件）"""
        self.logger.debug("清理本地缓存")
        try:
            cache_dir = os.path.join(self.app_dir, "cache")
            if os.path.exists(cache_dir):
                for filename in os.listdir(cache_dir):
                    if filename.endswith("_playlists.json") and not filename.startswith(self.username + "_"):
                        file_path = os.path.join(cache_dir, filename)
                        os.remove(file_path)
                        self.logger.debug(f"删除缓存文件: {file_path}")
        except Exception as e:
            self.logger.debug(f"清理本地缓存失败: {e}")

    def get_playlists(self):
        """获取所有歌单"""
        return self.playlists

    def get_playlist_songs(self, playlist_name):
        """获取指定歌单的歌曲列表"""
        if playlist_name in self.playlists:
            return self.playlists[playlist_name]
        return []

    # playlist_manager.py - 修改为使用父窗口的会话
    def get_session(self):
        """获取HTTP会话对象"""
        if self.parent and hasattr(self.parent, 'get_session'):
            return self.parent.get_session()
        return None

    # playlist_manager.py - 修改为使用父窗口的会话清理
    def close_session(self):
        """关闭HTTP会话"""
        try:
            # 不再直接管理session，由父窗口统一管理
            if self.parent and hasattr(self.parent, 'reset_session'):
                self.parent.reset_session()
            self.logger.debug("HTTP会话已通过父窗口关闭")
        except Exception as e:
            self.logger.debug(f"关闭HTTP会话时出错: {e}")

    def get_session_auth(self):
        """获取session认证信息"""
        if self.session:
            return self.session.auth
        return None

    def set_session_auth(self, username, password):
        """设置session认证信息"""
        if self.session:
            self.session.auth = HTTPBasicAuth(username, password)

    def get_session_headers(self):
        """获取session头部信息"""
        if self.session:
            return self.session.headers
        return None

    def add_session_header(self, key, value):
        """添加session头部信息"""
        if self.session:
            self.session.headers[key] = value

    # ===== 新增：根据文件夹结构创建歌单功能 =====
    def create_playlists_from_folders(self, music_files, enabled=True, force_create=False):
        """根据文件夹结构创建歌单"""
        if not enabled:
            return 0
        if not force_create and len(self.playlists["system"]) > 0:
            self.logger.debug(f"已有 {len(self.playlists['system'])} 个系统歌单，跳过自动创建")
            return 0
        folder_groups = {}
        for track in music_files:
            dir_path = os.path.dirname(track['path'])
            folder_name = os.path.basename(dir_path)
            if not folder_name or folder_name == '/':
                continue
            if folder_name not in folder_groups:
                folder_groups[folder_name] = []
            # 关键修复：存储完整track信息而不仅仅是歌名
            folder_groups[folder_name].append(track['name'])
        created_count = 0
        for folder_name, songs in folder_groups.items():
            if not songs:
                continue
            playlist_name = folder_name
            if playlist_name not in self.playlists["system"]:
                self.playlists["system"][playlist_name] = songs
                created_count += 1
                self.logger.debug(f"自动创建系统歌单: {playlist_name} ({len(songs)} 首歌曲)")
        if created_count > 0:
            self.save_playlists_local_cache()
            self.save_playlists()
        return created_count

    def load_playlist_settings(self):
        """加载歌单设置"""
        settings_file = os.path.join(self.app_dir, "playlist_settings.json")
        try:
            if os.path.exists(settings_file):
                with open(settings_file, 'r', encoding='utf-8') as f:
                    settings = json.load(f)
                    self.auto_create_playlists_from_folders = settings.get('auto_create_from_folders', True)
        except Exception as e:
            self.logger.debug(f"加载歌单设置失败: {e}")

    def save_playlist_settings(self):
        """保存歌单设置"""
        settings_file = os.path.join(self.app_dir, "playlist_settings.json")
        try:
            settings = {
                'auto_create_from_folders': self.auto_create_playlists_from_folders
            }
            with open(settings_file, 'w', encoding='utf-8') as f:
                json.dump(settings, f, ensure_ascii=False, indent=2)
        except Exception as e:
            self.logger.debug(f"保存歌单设置失败: {e}")

    def on_playlist_tab_changed(self, index):
        """处理歌单Tab切换 - 修正显示逻辑"""
        self.logger.debug(f"歌单Tab切换到: {index}")
        # 记录当前分类
        self.current_playlist_category = index
        # 触发父窗口更新歌单列表
        if self.parent and hasattr(self.parent, 'update_playlist_list'):
            self.parent.update_playlist_list()


    def update_playlist_list(self):
        """更新歌单列表 UI"""
        self.logger.debug("更新歌单列表")
        try:
            if not self.parent:
                return

            current_playlist_name = self.parent.current_playlist
            playlist_list = self.parent.playlist_list
            playlist_list.setUpdatesEnabled(False)
            try:
                playlist_list.clear()
                filter_keyword = getattr(self.parent, 'playlist_filter_keyword', '').strip().lower()
                current_tab_index = self.parent.playlist_tab_bar.currentIndex()
                playlists_to_show = []

                if current_tab_index == 0:  # 系统歌单
                    if "system" in self.playlists:
                        for playlist_name in self.playlists["system"].keys():
                            if not filter_keyword or filter_keyword in playlist_name.lower():
                                playlists_to_show.append(playlist_name)
                elif current_tab_index == 1:  # 自定义歌单 + 所有歌曲
                    if not filter_keyword or "所有歌曲".lower().find(filter_keyword) != -1:
                        playlists_to_show.append("所有歌曲")
                    if "custom" in self.playlists:
                        for playlist_name in self.playlists["custom"].keys():
                            if not filter_keyword or filter_keyword in playlist_name.lower():
                                playlists_to_show.append(playlist_name)

                # 渲染歌单列表
                for playlist_name in playlists_to_show:
                    item = QListWidgetItem(playlist_name)
                    if playlist_name == current_playlist_name:
                        item.setBackground(QColor(135, 206, 250))
                    playlist_list.addItem(item)

                # 保持当前选中状态
                if current_playlist_name in playlists_to_show:
                    items = playlist_list.findItems(current_playlist_name, Qt.MatchExactly)
                    if items:
                        playlist_list.currentRowChanged.disconnect(self.on_playlist_changed)
                        playlist_list.setCurrentItem(items[0])
                        playlist_list.currentRowChanged.connect(self.on_playlist_changed)

                self.logger.debug(f"歌单列表更新完成，共 {len(playlists_to_show)} 个歌单")
                # 新增：发送歌单数量更新信号
                self.update_playlist_count_signal.emit(len(playlists_to_show))
            finally:
                playlist_list.setUpdatesEnabled(True)
                playlist_list.update()
                QApplication.processEvents()
        except Exception as e:
            self.logger.debug(f"更新歌单列表时出错: {e}")
            import traceback
            traceback.print_exc()


    def update_playlist_list_no_switch(self):
        """更新歌单列表 UI（不触发切换）"""
        self.logger.debug("更新歌单列表（不触发切换）")
        try:
            if not self.parent:
                return

            current_playlist_name = self.parent.current_playlist
            playlist_list = self.parent.playlist_list
            playlist_list.setUpdatesEnabled(False)
            try:
                playlist_list.clear()
                filter_keyword = getattr(self.parent, 'playlist_filter_keyword', '').strip().lower()
                playlists_to_show = []

                current_tab_index = self.parent.playlist_tab_bar.currentIndex()
                if current_tab_index == 0:
                    if "system" in self.playlists:
                        for playlist_name in self.playlists["system"].keys():
                            if not filter_keyword or filter_keyword in playlist_name.lower():
                                playlists_to_show.append(playlist_name)
                else:
                    if not filter_keyword or "所有歌曲".lower().find(filter_keyword) != -1:
                        playlists_to_show.append("所有歌曲")
                    if "custom" in self.playlists:
                        for playlist_name in self.playlists["custom"].keys():
                            if not filter_keyword or filter_keyword in playlist_name.lower():
                                playlists_to_show.append(playlist_name)

                for playlist_name in playlists_to_show:
                    item = QListWidgetItem(playlist_name)
                    if playlist_name == current_playlist_name:
                        item.setBackground(QColor(135, 206, 250))
                        playlist_list.currentRowChanged.disconnect(self.parent.on_playlist_changed)
                        playlist_list.setCurrentItem(item)
                        playlist_list.currentRowChanged.connect(self.parent.on_playlist_changed)
                    playlist_list.addItem(item)

                self.logger.debug(f"歌单列表更新完成，共 {len(playlists_to_show)} 个歌单")
            finally:
                playlist_list.setUpdatesEnabled(True)
                playlist_list.update()
                QApplication.processEvents()
        except Exception as e:
            self.logger.debug(f"更新歌单列表时出错: {e}")
            import traceback
            traceback.print_exc()

    def on_playlist_changed(self, row):
        """歌单切换事件处理"""
        self.logger.debug(f"歌单切换事件，行: {row}")
        try:
            if not self.parent:
                return

            # 如果不允许切换歌单，直接返回
            if not self.allow_playlist_change:
                self.logger.debug("歌单切换被禁止")
                return

            # 修改1：通过信号获取多选状态
            is_multi_select = False
            if hasattr(self.parent, 'playlist_list'):
                selected_items = self.parent.playlist_list.selectedItems()
                is_multi_select = selected_items and len(selected_items) > 1

            if is_multi_select:
                self.logger.debug("检测到多选操作，不触发歌单切换")
                return

            if row >= 0:
                item = self.parent.playlist_list.item(row)
                if item:
                    playlist_name = item.text()
                    self.logger.info(f"切换到歌单: {playlist_name}")

                    # 更新当前歌单
                    self.parent.current_playlist = playlist_name
                    self.set_current_playlist_label_signal.emit(playlist_name)

                    # 修改2：通过信号通知清空搜索框
                    self.clear_search_box_signal.emit()  # 新增信号

                    # 更新文件列表
                    self.update_file_list_signal.emit()

                    # 重置当前播放索引
                    self.set_current_index_signal.emit(-1)
                    self.logger.debug(f"歌单切换完成，当前播放索引重置为: -1")

                    # 确保歌单切换后立即刷新界面
                    QTimer.singleShot(0, lambda: QApplication.processEvents())
                    self.logger.debug("已触发界面立即刷新")
                else:
                    self.logger.debug(f"歌单项不存在")
            else:
                self.logger.debug(f"无效的行号: {row}")
        except Exception as e:
            self.logger.debug(f"歌单切换事件处理时出错: {e}")
            import traceback
            traceback.print_exc()


