# ===== 新增：日志管理模块 =====
import os
import sys
from PyQt5.QtCore import Qt, QSize
from PyQt5.QtWidgets import (QVBoxLayout, QHBoxLayout,
                             QPushButton, QStyle, QFrame, QDialog)
from yc import LogManager

class ControlWindow(QDialog):
    """独立播放控制窗口"""

    def __init__(self, parent=None, logger=None, style_config=None):
        super().__init__(parent)
        self.logger = LogManager()
        # 添加窗口标识
        self.setObjectName("controlWindow")
        # 修改样式配置为统一文件
        self.style_config = style_config or {
            'css_paths': [
                "../css/app.css" # 从/yc/目录回到主目录的css文件夹
            ]
        }
        self.setWindowTitle("播放控制")
        self.setWindowFlags(Qt.Tool | Qt.WindowStaysOnTopHint | Qt.FramelessWindowHint)
        self.setAttribute(Qt.WA_TranslucentBackground)
        self.setFixedSize(300, 80)
        # 初始化UI
        self.init_ui()
        # 加载外部样式表
        self.load_external_stylesheet()
        # 添加拖拽功能
        self.drag_position = None

    def load_external_stylesheet(self):
        """加载外部样式表"""
        try:
            # 从配置中获取路径
            css_paths = self._get_css_paths()
            for css_path in css_paths:
                if os.path.exists(css_path):
                    with open(css_path, 'r', encoding='utf-8') as f:
                        stylesheet = f.read()
                    self.setStyleSheet(stylesheet)
                    self.logger.debug(f"成功加载控制窗口样式表: {css_path}")
                    return
            self.logger.debug("未找到控制窗口样式表文件")
        except Exception as e:
            self.logger.error(f"加载控制窗口样式表失败: {e}")

    def _get_css_paths(self):
        """获取样式表文件路径"""
        base_dir = os.path.dirname(os.path.abspath(__file__))
        exec_dir = os.path.dirname(sys.executable)
        paths = []
        for path in self.style_config['css_paths']:
            paths.append(os.path.join(base_dir, path))
            paths.append(os.path.join(exec_dir, path))
        return paths


    def init_ui(self):
        layout = QVBoxLayout()
        layout.setContentsMargins(0, 0, 0, 0)
        self.setLayout(layout)
        # 控制面板
        control_frame = QFrame()
        control_frame.setObjectName("control-frame")
        control_layout = QHBoxLayout(control_frame)
        control_layout.setSpacing(15)
        # 上一首按钮
        self.prev_button = QPushButton()
        self.prev_button.setObjectName("control-button")
        self.prev_button.setIcon(self.style().standardIcon(QStyle.SP_MediaSkipBackward))
        self.prev_button.setIconSize(QSize(24, 24))
        self.prev_button.setFixedSize(40, 40)
        control_layout.addWidget(self.prev_button)
        # 播放/暂停按钮
        self.play_button = QPushButton()
        self.play_button.setObjectName("play-button")
        self.play_button.setIcon(self.style().standardIcon(QStyle.SP_MediaPlay))
        self.play_button.setIconSize(QSize(28, 28))
        self.play_button.setFixedSize(50, 50)
        control_layout.addWidget(self.play_button)
        # 停止按钮
        self.stop_button = QPushButton()
        self.stop_button.setObjectName("control-button")
        self.stop_button.setIcon(self.style().standardIcon(QStyle.SP_MediaStop))
        self.stop_button.setIconSize(QSize(24, 24))
        self.stop_button.setFixedSize(40, 40)
        control_layout.addWidget(self.stop_button)
        # 下一首按钮
        self.next_button = QPushButton()
        self.next_button.setObjectName("control-button")
        self.next_button.setIcon(self.style().standardIcon(QStyle.SP_MediaSkipForward))
        self.next_button.setIconSize(QSize(24, 24))
        self.next_button.setFixedSize(40, 40)
        control_layout.addWidget(self.next_button)
        layout.addWidget(control_frame)

    def mousePressEvent(self, event):
        if event.button() == Qt.LeftButton:
            self.drag_position = event.globalPos() - self.frameGeometry().topLeft()
            event.accept()

    def mouseMoveEvent(self, event):
        if event.buttons() == Qt.LeftButton and self.drag_position:
            self.move(event.globalPos() - self.drag_position)
            event.accept()

    def mouseReleaseEvent(self, event):
        self.drag_position = None