# ===== 新增：日志管理模块 =====
from PyQt5.QtGui import QFont
from PyQt5.QtWidgets import (QVBoxLayout, QHBoxLayout,
                             QPushButton, QCheckBox, QDialog,
                             QTextEdit, QMessageBox)
import re
from yc import LogManager


class LyricDialog(QDialog):
    """歌词编辑对话框"""

    def __init__(self, parent=None, lyric_content="", enable_upload=False, logger=None):
        super().__init__(parent)
        self.logger = logger or  LogManager()
        self.setWindowTitle("歌词编辑")
        self.setModal(True)
        self.resize(600, 400)
        layout = QVBoxLayout()
        self.setLayout(layout)
        # 歌词编辑框
        self.lyric_edit = QTextEdit()
        self.lyric_edit.setPlaceholderText("请输入歌词内容，支持LRC格式：\n[mm:ss.xx]歌词内容")
        self.lyric_edit.setText(lyric_content)
        self.lyric_edit.setFont(QFont("Arial", 12))
        layout.addWidget(self.lyric_edit)
        # 上传到WebDAV选项
        if enable_upload:
            self.upload_checkbox = QCheckBox("同时上传到WebDAV服务器")
            self.upload_checkbox.setChecked(True)
            layout.addWidget(self.upload_checkbox)
        # 按钮
        button_layout = QHBoxLayout()
        self.save_button = QPushButton("保存")
        self.save_button.clicked.connect(self.accept)
        button_layout.addWidget(self.save_button)
        self.cancel_button = QPushButton("取消")
        self.cancel_button.clicked.connect(self.reject)
        button_layout.addWidget(self.cancel_button)
        layout.addLayout(button_layout)

    def get_lyric_content(self):
        """获取歌词内容"""
        content = self.lyric_edit.toPlainText()
        self.logger.debug(f"获取歌词内容，长度: {len(content)}")
        # 验证LRC格式
        if content and not self._validate_lrc_format(content):
            self.logger.warning("歌词内容可能不符合LRC格式")
        return content

    def should_upload_to_webdav(self):
        """是否需要上传到WebDAV"""
        should_upload = hasattr(self, 'upload_checkbox') and self.upload_checkbox.isChecked()
        self.logger.debug(f"上传到WebDAV设置: {should_upload}")
        return should_upload

    def _validate_lrc_format(self, content):
        """简单的LRC格式验证"""
        # LRC格式示例: [mm:ss.xx]歌词内容
        lrc_pattern = re.compile(r'^\[\d{2}:\d{2}\.\d{2}\].+$', re.MULTILINE)
        return bool(lrc_pattern.search(content))

    def accept(self):
        """保存歌词"""
        self.logger.info("用户保存歌词内容")
        self._content = self.get_lyric_content()
        # 如果内容不为空但格式不正确，显示警告
        if  self._content and not self._validate_lrc_format( self._content):
            reply = QMessageBox.question(
                self, "格式警告",
                "歌词内容可能不符合LRC格式，是否继续保存？",
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No
            )
            if reply == QMessageBox.No:
                return
        super().accept()

    # 添加获取保存内容的方法
    def get_saved_content(self):
        """获取已保存的歌词内容"""
        return getattr(self, '_content', None)

    def reject(self):
        """取消编辑"""
        self.logger.info("用户取消歌词编辑")
        super().reject()