# ===== 新增：日志管理模块 =====
from PyQt5.QtGui import QFont, QColor, QSyntaxHighlighter, \
    QTextCharFormat
from PyQt5.QtCore import QTimer
from yc import LogManager


class LyricHighlighter(QSyntaxHighlighter):
    """歌词高亮器"""

    def __init__(self, parent, logger=None, config=None):
        super().__init__(parent)
        self.logger = logger or  LogManager()
        self.config = config or {
            'font_family': 'Arial',
            'font_size': 14,
            'font_weight': QFont.Bold,
            'background_color': QColor(135, 206, 250)  # LightSkyBlue
        }
        self.current_line_format = QTextCharFormat()
        self._setup_format()
        self.current_line = -1
        self.last_highlighted = -1
        self._highlight_timer = None

    def _setup_format(self):
        """设置高亮格式"""
        font = QFont(
            self.config['font_family'],
            self.config['font_size'],
            self.config['font_weight']
        )
        self.current_line_format.setFont(font)
        self.current_line_format.setBackground(QColor(0, 0, 0, 0))  # <--- 改为这一行
        self.current_line_format.setFontWeight(self.config['font_weight'])

    def set_current_line(self, line):
        """设置当前行（带防抖）"""
        if line != self.current_line:
            self.current_line = line
            self.logger.debug(f"设置当前高亮行: {line}")
            # 防抖处理，避免频繁重绘
            if self._highlight_timer:
                self._highlight_timer.stop()
            self._highlight_timer = QTimer()
            self._highlight_timer.setSingleShot(True)
            self._highlight_timer.timeout.connect(self._do_highlight)
            self._highlight_timer.start(100)  # 100ms延迟

    def _do_highlight(self):
        """执行高亮操作"""
        if self.current_line != self.last_highlighted:
            self.last_highlighted = self.current_line
            self.logger.debug(f"重新高亮行: {self.current_line}")
            self.rehighlight()

    def highlightBlock(self, text):
        """重写高亮块方法，应用格式到文本块"""
        # 获取当前块的行号
        current_block = self.currentBlock()
        block_number = current_block.blockNumber()
        # 如果当前块是需要高亮的行，应用格式
        if block_number == self.current_line:
            self.setFormat(0, len(text), self.current_line_format)
            self.logger.debug(f"高亮行 {block_number}")