# ===== 新增：日志管理模块 =====
import os
import tempfile
from PyQt5.QtCore import QThread, pyqtSignal, QMutex, QMutexLocker, QTimer
from urllib.parse import quote
from yc import LogManager


class LyricSyncWorker(QThread):
    """歌词同步工作线程"""
    # 在类开头定义信号
    progress = pyqtSignal(str)  # 进度信号
    finished = pyqtSignal(str, bool)  # 完成信号 (消息, 是否成功)
    error = pyqtSignal(str)  # 错误信号

    def __init__(self, player, song_name, song_path, asr_server_url, asr_api_key, logger=None):
        super().__init__()
        self.logger = logger or LogManager()
        self.player = player
        self.song_name = song_name
        self.song_path = song_path
        self.asr_server_url = asr_server_url
        self.asr_api_key = asr_api_key
        self._is_running = True
        self._mutex = QMutex()
        # 创建ASR客户端
        from .speech_recognition_client import SpeechRecognitionClient
        self.asr_client = SpeechRecognitionClient(
            server_url=asr_server_url,
            api_key=asr_api_key,
            logger=self.logger
        )

    def run(self):
        """执行歌词同步"""
        try:
            if not self._check_prerequisites():
                return
            tmp_file_path = None
            try:
                tmp_file_path = self._download_song()
                if not tmp_file_path:
                    return
                result = self._transcribe_audio(tmp_file_path)
                if not result:
                    return
                lyric_content = self._convert_to_lrc(result)
                if not lyric_content:
                    return
                self._save_lyric(lyric_content)
            finally:
                self._cleanup_temp_file(tmp_file_path)
        except Exception as e:
            self.logger.error(f"在线同步歌词时出错: {e}")
            import traceback
            traceback.print_exc()
            self._safe_emit_error("在线同步歌词失败")

    def _check_prerequisites(self):
        """检查前置条件"""
        with QMutexLocker(self._mutex):
            if not self._is_running:
                return False
        if not self.player.playing or not self.player.current_track:
            self._safe_emit_error("请先播放一首歌曲")
            return False
        if not self.asr_server_url or not self.asr_api_key:
            self._safe_emit_error("请先在设置中配置ASR服务器地址和API Key")
            return False
        if not self.song_path:
            self._safe_emit_error("无法获取当前播放歌曲的路径")
            return False
        return True

    def _download_song(self):
        """下载歌曲到临时文件"""
        with QMutexLocker(self._mutex):
            if not self._is_running:
                return None
        self._safe_emit_progress("正在下载歌曲...")
        try:
            with tempfile.NamedTemporaryFile(suffix=os.path.splitext(self.song_name)[1], delete=False) as tmp_file:
                tmp_file_path = tmp_file.name
            self.player._download_single_file(self.song_path, tmp_file_path)
            self.logger.debug(f"歌曲已下载到临时文件: {tmp_file_path}")
            return tmp_file_path
        except Exception as e:
            self.logger.error(f"下载歌曲失败: {e}")
            self._safe_emit_error("下载歌曲失败")
            return None

    def _download_single_file(self, webdav_path, save_path):
        """下载单个文件"""
        self.logger.debug(f"下载单个文件: {webdav_path} -> {save_path}")
        # 构建下载URL
        protocol = self.webdav_url.split('://')[0]
        host = self.webdav_url.split('://')[1].split('/')[0]
        base_webdav = f"{protocol}://{host}"
        encoded_username = quote(self.username)
        encoded_password = quote(self.password)
        download_url = f"{base_webdav}{webdav_path}"
        auth_url = f"{protocol}://{encoded_username}:{encoded_password}@{host}{webdav_path}"
        self.logger.debug(f"下载URL: {download_url}")
        self.logger.debug(f"认证URL: {auth_url}")
        # 确保会话已初始化
        if not self.init_session():
            raise Exception("无法初始化HTTP会话")
        # 下载文件
        response = self.session.get(download_url, auth=(self.username, self.password), stream=True, timeout=30)
        if response.status_code != 200:
            raise Exception(f"下载失败，服务器返回状态码: {response.status_code}")
        # 保存文件
        with open(save_path, 'wb') as f:
            for chunk in response.iter_content(chunk_size=8192):
                if chunk:
                    f.write(chunk)
        self.logger.debug(f"文件下载成功: {save_path}")

    def _transcribe_audio(self, file_path):
        """识别音频文件"""
        with QMutexLocker(self._mutex):
            if not self._is_running:
                return None
        self._safe_emit_progress("正在识别歌词...")
        try:
            result = self.asr_client.transcribe_file(
                file_path=file_path,
                response_format="json"
            )
            self.logger.debug(f"ASR返回结果: {result}")
            return result
        except Exception as e:
            self.logger.error(f"ASR请求失败: {e}")
            # 检查是否是服务器错误
            error_str = str(e)
            if "500" in error_str or "internal_error" in error_str:
                self._safe_emit_error("ASR服务器内部错误，请稍后重试")
            else:
                self._safe_emit_error("歌词识别失败")
            return None

    def _convert_to_lrc(self, result):
        """将JSON数据转换为LRC格式"""
        with QMutexLocker(self._mutex):
            if not self._is_running:
                return None
        self._safe_emit_progress("正在转换歌词格式...")
        try:
            lyric_content = self.convert_json_to_lrc(result)
            return lyric_content
        except Exception as e:
            self.logger.error(f"转换歌词格式失败: {e}")
            self._safe_emit_error("转换歌词格式失败")
            return None

    def convert_json_to_lrc(self, json_data):
        """
        将JSON格式的歌词数据转换为LRC格式
        Args:
            json_data: ASR服务器返回的JSON数据
        Returns:
            LRC格式的歌词字符串
        """
        try:
            # 检查是否有segments字段
            if 'segments' not in json_data:
                self.logger.debug("JSON数据中没有segments字段")
                return ""
            lrc_lines = []
            # 处理每个歌词片段
            for segment in json_data['segments']:
                start_time = segment.get('start', 0)
                text = segment.get('text', '').strip()
                # 跳过空文本
                if not text:
                    continue
                # 将秒转换为LRC时间格式 [mm:ss.xx]
                minutes = int(start_time // 60)
                seconds = int(start_time % 60)
                centiseconds = int((start_time % 1) * 100)
                # 格式化时间字符串
                time_str = f"[{minutes:02d}:{seconds:02d}.{centiseconds:02d}]"
                # 添加到歌词列表
                lrc_lines.append(f"{time_str}{text}")
            # 合并所有歌词行
            lrc_content = "\n".join(lrc_lines)
            self.logger.debug(f"转换后的LRC歌词行数: {len(lrc_lines)}")
            return lrc_content
        except Exception as e:
            self.logger.debug(f"转换JSON到LRC失败: {e}")
            import traceback
            traceback.print_exc()
            return ""


    def validate_lrc_content(self, lrc_content):
        """
        验证LRC歌词内容
        Args:
            lrc_content: LRC格式的歌词字符串
        Returns:
            bool: 歌词是否有效
        """
        try:
            if not lrc_content or not lrc_content.strip():
                self.logger.debug("LRC内容为空")
                return False
            # 检查是否包含时间标签
            lines = lrc_content.strip().split('\n')
            time_line_count = 0
            for line in lines:
                if '[' in line and ']' in line:
                    time_line_count += 1
            # 至少要有50%的行包含时间标签
            if time_line_count < len(lines) * 0.5:
                self.logger.debug(f"时间标签行数不足: {time_line_count}/{len(lines)}")
                return False
            self.logger.debug(f"LRC验证通过: {time_line_count}/{len(lines)} 行包含时间标签")
            return True
        except Exception as e:
            self.logger.debug(f"验证LRC内容失败: {e}")
            return False

    def _save_lyric(self, lyric_content):
        """保存歌词到WebDAV服务器"""
        with QMutexLocker(self._mutex):
            if not self._is_running:
                return
        # 验证生成的LRC内容
        try:
            if not self.validate_lrc_content(lyric_content):
                self._safe_emit_error("生成的歌词格式不正确")
                return
        except Exception as e:
            self.logger.error(f"验证歌词内容失败: {e}")
            self._safe_emit_error("验证歌词内容失败")
            return
        # 上传歌词到WebDAV
        self._safe_emit_progress("正在保存歌词...")
        try:
            self.player.upload_lyric_to_webdav(self.song_name, lyric_content)
            # 更新歌词文件路径
            song_dir = os.path.dirname(self.song_path)
            song_filename = os.path.splitext(self.song_name)[0]
            self.logger.debug(f"song_dir:{song_dir}")
            self.logger.debug(f"song_filename:{song_filename}")
            self.player.lyric_file = f"{song_dir}/{song_filename}.lrc"
            # 发送成功信号
            self._safe_emit_finished("歌词同步成功", True)
        except Exception as e:
            self.logger.error(f"保存歌词到WebDAV失败: {e}")
            self._safe_emit_finished("歌词识别成功，但保存到服务器失败", False)

    def _cleanup_temp_file(self, tmp_file_path):
        """清理临时文件"""
        if tmp_file_path and os.path.exists(tmp_file_path):
            try:
                os.remove(tmp_file_path)
                self.logger.debug(f"临时文件已删除: {tmp_file_path}")
            except Exception as e:
                self.logger.error(f"删除临时文件失败: {e}")

    def _safe_emit_progress(self, message):
        """安全地发射进度信号"""
        try:
            with QMutexLocker(self._mutex):
                if self._is_running:
                    self.progress.emit(message)
        except Exception as e:
            self.logger.error(f"发射进度信号时出错: {e}")

    def _safe_emit_finished(self, message, success):
        """安全地发射完成信号"""
        try:
            with QMutexLocker(self._mutex):
                if self._is_running:
                    self.finished.emit(message, success)
        except Exception as e:
            self.logger.error(f"发射完成信号时出错: {e}")

    def _safe_emit_error(self, error_message):
        """安全地发射错误信号"""
        try:
            with QMutexLocker(self._mutex):
                if self._is_running:
                    self.error.emit(error_message)
        except Exception as e:
            self.logger.error(f"发射错误信号时出错: {e}")

    def stop(self):
        """安全停止线程"""
        with QMutexLocker(self._mutex):
            self._is_running = False
        # 等待线程自然结束
        if not self.wait(5000):  # 等待5秒
            self.logger.warning("线程未能在指定时间内结束，强制终止")
            self.terminate()
            self.wait()