# ===== 新增：日志管理模块 =====
import os
import sys
from PyQt5.QtCore import Qt, QSize
from PyQt5.QtGui import QFont, QKeySequence
from PyQt5.QtWidgets import (QVBoxLayout, QHBoxLayout,
                             QPushButton, QSlider, QLabel, QStyle, QFrame, QComboBox, QDialog,
                             QTextEdit, QShortcut)
from yc import LogManager


class LyricWindow(QDialog):
    """独立歌词显示窗口（整合播放控制功能）"""

    def __init__(self, parent=None, logger=None, style_config=None):
        super().__init__(parent)
        self.logger = logger or  LogManager()
        # 添加窗口标识
        self.setObjectName("lyricWindow")
        # 修改样式配置为统一文件
        self.style_config = style_config or {
            'css_paths': [
                "../css/app.css" # 从/yc/目录回到主目录的css文件夹
            ]
        }
        self.setWindowTitle("精简模式")
        self.setWindowFlags(Qt.Tool | Qt.WindowStaysOnTopHint | Qt.FramelessWindowHint)
        self.setAttribute(Qt.WA_TranslucentBackground)
        self.setFixedSize(400, 700)  # 增加高度以容纳进度条
        # 添加进度更新相关属性
        self.current_position = 0
        self.total_duration = 0
        # 初始化UI
        self.init_ui()
        # 加载外部样式表
        self.load_external_stylesheet()
        # 添加拖拽功能
        self.drag_position = None
        # 关闭事件处理
        self.finished.connect(self.on_window_closed)
        # 设置快捷键
        self.shortcut_toggle = QShortcut(QKeySequence("Ctrl+Alt+L"), self)
        self.shortcut_toggle.activated.connect(self.toggle_lyric_window)

    def load_external_stylesheet(self):
        """加载外部样式表"""
        try:
            # 从配置中获取路径
            css_paths = self._get_css_paths()
            for css_path in css_paths:
                if os.path.exists(css_path):
                    with open(css_path, 'r', encoding='utf-8') as f:
                        stylesheet = f.read()
                    self.setStyleSheet(stylesheet)
                    self.logger.debug(f"成功加载歌词窗口样式表: {css_path}")
                    return
            self.logger.debug("未找到歌词窗口样式表文件")
        except Exception as e:
            self.logger.error(f"加载歌词窗口样式表失败: {e}")

    def _get_css_paths(self):
        """获取样式表文件路径"""
        base_dir = os.path.dirname(os.path.abspath(__file__))
        exec_dir = os.path.dirname(sys.executable)
        paths = []
        for path in self.style_config['css_paths']:
            paths.append(os.path.join(base_dir, path))
            paths.append(os.path.join(exec_dir, path))
        return paths

    def on_window_closed(self):
        """窗口关闭时的处理"""
        self.logger.debug("LyricWindow closed")
        if self.parent():
            try:
                self.parent().on_lyric_window_closed()
            except Exception as e:
                self.logger.error(f"调用父窗口on_lyric_window_closed失败: {e}")

    def toggle_lyric_window(self):
        """切换精简模式"""
        if self.parent():
            try:
                self.parent().toggle_lyric_window()
            except Exception as e:
                self.logger.error(f"调用父窗口toggle_lyric_window失败: {e}")

    def init_ui(self):
        layout = QVBoxLayout()
        layout.setContentsMargins(0, 0, 0, 0)
        self.setLayout(layout)
        # 标题栏（用于拖拽）
        title_bar = QFrame()
        title_bar.setFixedHeight(40)  # 增加高度以适应更大的字体
        title_bar.setObjectName("title-bar")
        title_layout = QHBoxLayout(title_bar)
        title_layout.setContentsMargins(10, 0, 10, 0)
        # 当前播放歌曲标签 - 字体更大
        self.current_track_label = QLabel("未播放")
        self.current_track_label.setObjectName("current-track-label")
        self.current_track_label.setFont(QFont("Arial", 14, QFont.Bold))  # 字体大小14，加粗
        self.current_track_label.setAlignment(Qt.AlignCenter)
        title_layout.addWidget(self.current_track_label)
        title_label = QLabel("精简模式")
        title_layout.addWidget(title_label)
        title_layout.addStretch()
        close_btn = QPushButton("×")
        close_btn.setObjectName("close-button")
        close_btn.setFixedSize(20, 20)
        close_btn.clicked.connect(self.close)
        title_layout.addWidget(close_btn)
        layout.addWidget(title_bar)
        # 歌词显示区域
        self.lyric_display = QTextEdit()
        self.lyric_display.setObjectName("lyric-display")
        self.lyric_display.setReadOnly(True)
        self.lyric_display.setFont(QFont("Arial", 14))
        self.lyric_display.setAlignment(Qt.AlignCenter)
        layout.addWidget(self.lyric_display)
        # 播放控制区域
        control_frame = QFrame()
        control_frame.setObjectName("control-frame")
        control_layout = QVBoxLayout(control_frame)
        control_layout.setContentsMargins(10, 10, 10, 10)
        # 播放进度控制
        progress_layout = QHBoxLayout()
        progress_layout.setContentsMargins(0, 0, 0, 0)
        # 当前时间标签
        self.current_time_label = QLabel("00:00")
        self.current_time_label.setObjectName("time-label")
        self.current_time_label.setFont(QFont("Arial", 9))
        self.current_time_label.setMinimumWidth(40)
        progress_layout.addWidget(self.current_time_label)
        # 进度条
        self.position_slider = QSlider(Qt.Horizontal)
        self.position_slider.setObjectName("progress-slider")
        self.position_slider.setRange(0, 100)
        self.position_slider.sliderMoved.connect(self.set_position)
        progress_layout.addWidget(self.position_slider)
        # 总时间标签
        self.total_time_label = QLabel("00:00")
        self.total_time_label.setObjectName("time-label")
        self.total_time_label.setFont(QFont("Arial", 9))
        self.total_time_label.setMinimumWidth(40)
        progress_layout.addWidget(self.total_time_label)
        control_layout.addLayout(progress_layout)
        # 播放控制按钮
        button_layout = QHBoxLayout()
        button_layout.setContentsMargins(0, 5, 0, 0)
        # 上一首按钮
        self.prev_button = QPushButton()
        self.prev_button.setObjectName("control-button")
        self.prev_button.setIcon(self.style().standardIcon(QStyle.SP_MediaSkipBackward))
        self.prev_button.setIconSize(QSize(24, 24))
        self.prev_button.setFixedSize(40, 40)
        button_layout.addWidget(self.prev_button)
        # 播放/暂停按钮
        self.play_button = QPushButton()
        self.play_button.setObjectName("play-button")
        self.play_button.setIcon(self.style().standardIcon(QStyle.SP_MediaPlay))
        self.play_button.setIconSize(QSize(28, 28))
        self.play_button.setFixedSize(50, 50)
        button_layout.addWidget(self.play_button)
        # 停止按钮
        self.stop_button = QPushButton()
        self.stop_button.setObjectName("control-button")
        self.stop_button.setIcon(self.style().standardIcon(QStyle.SP_MediaStop))
        self.stop_button.setIconSize(QSize(24, 24))
        self.stop_button.setFixedSize(40, 40)
        button_layout.addWidget(self.stop_button)
        # 下一首按钮
        self.next_button = QPushButton()
        self.next_button.setObjectName("control-button")
        self.next_button.setIcon(self.style().standardIcon(QStyle.SP_MediaSkipForward))
        self.next_button.setIconSize(QSize(24, 24))
        self.next_button.setFixedSize(40, 40)
        button_layout.addWidget(self.next_button)
        control_layout.addLayout(button_layout)
        # 播放模式选择
        mode_layout = QHBoxLayout()
        mode_layout.addWidget(QLabel("模式:"))
        self.mode_combo = QComboBox()
        self.mode_combo.setObjectName("mode-combo")
        self.mode_combo.addItems(["顺序播放", "随机播放", "循环播放", "单曲循环"])
        self.mode_combo.setMaximumWidth(100)
        mode_layout.addWidget(self.mode_combo)
        control_layout.addLayout(mode_layout)
        layout.addWidget(control_frame)

    def update_position(self, position):
        """更新播放位置（秒）"""
        try:
            self.current_position = position
            self.position_slider.setValue(int(position))
            self.current_time_label.setText(self._format_time_default(position))
            self.logger.debug(f"精简窗口更新播放位置: {position}s")
        except Exception as e:
            self.logger.error(f"更新播放位置失败: {e}")

    def update_duration(self, duration):
        """更新总时长（秒）"""
        try:
            self.total_duration = duration
            self.position_slider.setRange(0, int(duration))
            self.total_time_label.setText(self._format_time_default(duration))
        except Exception as e:
            self.logger.error(f"更新总时长失败: {e}")

    def update_playback_state(self, playing, paused):
        """更新播放状态"""
        try:
            if playing:
                if paused:
                    icon = self.style().standardIcon(QStyle.SP_MediaPlay)
                else:
                    icon = self.style().standardIcon(QStyle.SP_MediaPause)
            else:
                icon = self.style().standardIcon(QStyle.SP_MediaPlay)

            self.play_button.setIcon(icon)
        except Exception as e:
            self.logger.error(f"更新播放状态失败: {e}")

    def _format_time_default(self, seconds):
        """默认时间格式化方法"""
        minutes, seconds = divmod(int(seconds), 60)
        return f"{minutes:02d}:{seconds:02d}"


    def set_position(self, position):
        """设置播放位置"""
        try:
            if self.parent() and hasattr(self.parent(), 'set_position'):
                self.parent().set_position(position)
            else:
                self.logger.warning("父窗口不存在或缺少set_position方法")
        except Exception as e:
            self.logger.error(f"设置播放位置失败: {e}")

    def update_progress_info(self, current_time, total_time, position):
        """更新进度信息"""
        try:
            # 安全地调用父窗口方法
            if self.parent() and hasattr(self.parent(), 'format_time'):
                current_str = self.parent().format_time(current_time)
                total_str = self.parent().format_time(total_time)
            else:
                # 提供默认格式化方法
                current_str = self._format_time_default(current_time)
                total_str = self._format_time_default(total_time)
            self.current_time_label.setText(current_str)
            self.total_time_label.setText(total_str)
            if total_time > 0:
                self.position_slider.setRange(0, int(total_time))
                self.position_slider.setValue(int(position))
        except Exception as e:
            self.logger.error(f"更新进度信息失败: {e}")



    def mousePressEvent(self, event):
        if event.button() == Qt.LeftButton:
            self.drag_position = event.globalPos() - self.frameGeometry().topLeft()
            event.accept()

    def mouseMoveEvent(self, event):
        if event.buttons() == Qt.LeftButton and self.drag_position:
            self.move(event.globalPos() - self.drag_position)
            event.accept()

    def mouseReleaseEvent(self, event):
        self.drag_position = None

    def update_current_track(self, track_name):
        """更新当前播放歌曲显示"""
        try:
            if hasattr(self, 'current_track_label') and self.current_track_label:
                self.current_track_label.setText(track_name)
                self.logger.debug(f"更新当前播放歌曲: {track_name}")
        except Exception as e:
            self.logger.error(f"更新当前播放歌曲失败: {e}")