# ===== 新增：日志管理模块 =====
from PyQt5.QtWidgets import (QVBoxLayout, QLabel, QMessageBox, QDialog,
                             QLineEdit, QDialogButtonBox, QFormLayout, QHBoxLayout, QPushButton, QCheckBox)
from PyQt5.QtCore import Qt
from yc import LogManager


class SettingsDialog(QDialog):
    def __init__(self, parent=None, current_settings=None, logger=None):
        super().__init__(parent)
        self.logger = logger or LogManager()
        self.setWindowTitle("WebDAV 设置")
        self.setModal(True)
        self.resize(500, 300)  # 增加高度以容纳新设置
        # 默认设置
        default_settings = {
            'url': 'http://gdsy.mycloudnas.com:5007/music',
            'username': 'admin',
            'password': 'password',
            'asr_server_url': 'http://192.168.168.226:5000',
            'asr_api_key': 'token1',
            # 新增：自动创建歌单设置
            'auto_create_playlists': True
        }
        # 使用当前设置或默认设置
        self.settings = current_settings if current_settings else default_settings
        # 创建UI
        layout = QVBoxLayout()
        # 表单布局
        form_layout = QFormLayout()
        # WebDAV设置
        self.url_edit = QLineEdit(self.settings['url'])
        self.url_edit.setPlaceholderText("请输入WebDAV服务器地址")
        form_layout.addRow("服务器地址:", self.url_edit)
        self.username_edit = QLineEdit(self.settings['username'])
        self.username_edit.setPlaceholderText("请输入用户名")
        form_layout.addRow("用户名:", self.username_edit)
        # 密码输入框和显示/隐藏按钮
        password_layout = QHBoxLayout()
        self.password_edit = QLineEdit(self.settings['password'])
        self.password_edit.setEchoMode(QLineEdit.Password)
        self.password_edit.setPlaceholderText("请输入密码")
        password_layout.addWidget(self.password_edit)
        self.password_toggle = QPushButton("👁")
        self.password_toggle.setFixedSize(20, 20)
        self.password_toggle.setCheckable(True)
        self.password_toggle.toggled.connect(self._toggle_password_visibility)
        password_layout.addWidget(self.password_toggle)
        form_layout.addRow("密码:", password_layout)
        # ASR设置
        form_layout.addRow(QLabel(""))  # 添加分隔符
        self.asr_server_url_edit = QLineEdit(self.settings['asr_server_url'])
        self.asr_server_url_edit.setPlaceholderText("请输入ASR服务器地址")
        form_layout.addRow("ASR服务器地址:", self.asr_server_url_edit)
        # ASR API Key输入框和显示/隐藏按钮
        asr_key_layout = QHBoxLayout()
        self.asr_api_key_edit = QLineEdit(self.settings['asr_api_key'])
        self.asr_api_key_edit.setEchoMode(QLineEdit.Password)
        self.asr_api_key_edit.setPlaceholderText("请输入ASR API Key")
        asr_key_layout.addWidget(self.asr_api_key_edit)
        self.asr_key_toggle = QPushButton("👁")
        self.asr_key_toggle.setFixedSize(20, 20)
        self.asr_key_toggle.setCheckable(True)
        self.asr_key_toggle.toggled.connect(self._toggle_asr_key_visibility)
        asr_key_layout.addWidget(self.asr_key_toggle)
        form_layout.addRow("ASR API Key:", asr_key_layout)
        # 新增：歌单设置
        form_layout.addRow(QLabel(""))  # 添加分隔符
        playlist_label = QLabel("歌单设置")
        playlist_label.setStyleSheet("font-weight: bold; color: #2c3e50;")
        form_layout.addRow(playlist_label)
        # 自动创建歌单选项
        self.auto_create_checkbox = QCheckBox("根据文件夹结构自动创建歌单")
        self.auto_create_checkbox.setChecked(self.settings.get('auto_create_playlists', True))
        self.auto_create_checkbox.setToolTip("首次扫描服务器时，如果没有歌单，将根据文件夹结构自动创建歌单")
        form_layout.addRow("", self.auto_create_checkbox)
        # 添加说明文本
        info_label = QLabel("说明：仅在用户没有歌单（除了'所有歌曲'）时生效")
        info_label.setStyleSheet("color: #7f8c8d; font-size: 11px;")
        info_label.setWordWrap(True)
        form_layout.addRow("", info_label)
        layout.addLayout(form_layout)
        # 按钮
        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        self.setLayout(layout)

    def _toggle_password_visibility(self, checked):
        """切换密码可见性"""
        self.password_edit.setEchoMode(
            QLineEdit.Normal if checked else QLineEdit.Password
        )

    def _toggle_asr_key_visibility(self, checked):
        """切换ASR API Key可见性"""
        self.asr_api_key_edit.setEchoMode(
            QLineEdit.Normal if checked else QLineEdit.Password
        )

    def get_settings(self):
        """获取设置"""
        return {
            'url': self.url_edit.text().strip(),
            'username': self.username_edit.text().strip(),
            'password': self.password_edit.text().strip(),
            'asr_server_url': self.asr_server_url_edit.text().strip(),
            'asr_api_key': self.asr_api_key_edit.text().strip(),
            # 新增：自动创建歌单设置
            'auto_create_playlists': self.auto_create_checkbox.isChecked()
        }

    def accept(self):
        """确认设置"""
        self.logger.info("用户尝试保存设置")
        if not self._validate_settings():
            return
        self.logger.info("设置验证通过，保存设置")
        super().accept()

    def _validate_settings(self):
        """验证设置"""
        settings = self.get_settings()
        # 验证WebDAV设置
        if not self._validate_webdav_settings(settings):
            return False
        # 验证ASR设置
        if not self._validate_asr_settings(settings):
            return False
        # 确认HTTP使用
        if not self._confirm_http_usage(settings):
            return False
        return True

    def _validate_webdav_settings(self, settings):
        """验证WebDAV设置"""
        if not settings['url']:
            QMessageBox.warning(self, "警告", "请输入服务器地址")
            return False
        if not settings['url'].startswith(('http://', 'https://')):
            QMessageBox.warning(self, "警告", "服务器地址必须以 http:// 或 https:// 开头")
            return False
        if not settings['username']:
            QMessageBox.warning(self, "警告", "请输入用户名")
            return False
        if not settings['password']:
            QMessageBox.warning(self, "警告", "请输入密码")
            return False
        return True

    def _validate_asr_settings(self, settings):
        """验证ASR设置"""
        if not settings['asr_server_url']:
            QMessageBox.warning(self, "警告", "请输入ASR服务器地址")
            return False
        if not settings['asr_server_url'].startswith(('http://', 'https://')):
            QMessageBox.warning(self, "警告", "ASR服务器地址必须以 http:// 或 https:// 开头")
            return False
        if not settings['asr_api_key']:
            QMessageBox.warning(self, "警告", "请输入ASR API Key")
            return False
        return True

    def _confirm_http_usage(self, settings):
        """确认HTTP使用"""
        if not settings['url'].startswith('https://'):
            reply = QMessageBox.question(
                self, "安全警告",
                "您正在使用HTTP连接，凭据将以明文传输。\n"
                "强烈建议使用HTTPS连接。\n\n"
                "是否继续？",
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No
            )
            if reply == QMessageBox.No:
                return False
        return True