# ===== 新增：日志管理模块 =====
import os
import re
from typing import Dict, Any
import requests
from yc import LogManager


class SpeechRecognitionClient:
    """这是一个语音识别客户端方法，用于将音频文件发送到语音识别API服务，获取语音转文字的结果"""

    def __init__(self, server_url: str, api_key: str, logger=None, timeout=300):
        self.server_url = server_url.rstrip('/')
        self.api_key = api_key
        self.logger = logger or LogManager()
        self.timeout = timeout
        self.headers = {
            'Authorization': f'Bearer {api_key}'
        }

    def transcribe_file(
            self,
            file_path: str,
            model: str = 'large-v3',
            language: str = 'zh',
            response_format: str = 'json',
            temperature: float = 0.0
    ) -> Dict[str, Any]:
        """
        上传音频文件进行识别
        Args:
            file_path: 音频文件路径
            model: 使用的模型
            language: 语言代码
            response_format: 输出格式
            temperature: 采样温度
        Returns:
            识别结果字典
        """
        self.logger.info(f"开始识别音频文件: {file_path}")
        if not os.path.exists(file_path):
            self.logger.error(f"文件不存在: {file_path}")
            raise FileNotFoundError(f"File not found: {file_path}")
        url = f"{self.server_url}/v1/audio/transcriptions"
        try:
            with open(file_path, 'rb') as f:
                files = {'file': (os.path.basename(file_path), f, 'audio/*')}
                data = {
                    'model': model,
                    'language': language,
                    'response_format': response_format,
                    'temperature': temperature
                }
                self.logger.debug(f"发送请求到: {url}")
                response = requests.post(
                    url,
                    headers=self.headers,
                    files=files,
                    data=data,
                    timeout=self.timeout
                )
                self.logger.debug(f"响应状态码: {response.status_code}")
                if response.status_code != 200:
                    self._handle_error_response(response)
                    return {}
                # 解析响应内容
                try:
                    result = response.json()
                    self.logger.debug("响应内容解析成功")
                    return result
                except Exception as e:
                    self.logger.error(f"解析响应内容失败: {e}")
                    self.logger.debug(f"响应内容: {response.text}")
                    raise Exception("Failed to parse response content")
        except requests.exceptions.Timeout:
            self.logger.error("请求超时")
            raise Exception("请求超时，请检查网络连接或稍后重试")
        except requests.exceptions.ConnectionError as e:
            self.logger.error(f"网络连接错误: {e}")
            raise Exception("网络连接错误，请检查网络连接")
        except Exception as e:
            self.logger.error(f"请求过程中发生错误: {e}")
            raise

    def _handle_error_response(self, response):
        """处理错误响应"""
        try:
            error_data = response.json()
            error_message = error_data.get('error', {}).get('message', 'Unknown error')
            # 脱敏处理错误信息
            sanitized_message = self._sanitize_error_message(error_message)
            self.logger.error(f"API请求失败: {response.status_code} - {sanitized_message}")
            raise Exception(f"Request failed with status {response.status_code}: {sanitized_message}")
        except:
            self.logger.error(f"无法解析错误响应: {response.status_code}")
            raise Exception(f"Request failed with status {response.status_code}")

    def _sanitize_error_message(self, message):
        """脱敏处理错误信息中的敏感内容"""
        # 移除可能的API密钥、令牌等敏感信息
        patterns = [
            r'api_key=[a-zA-Z0-9\-_]+',
            r'token=[a-zA-Z0-9\-_]+',
            r'Bearer [a-zA-Z0-9\-_.]+'
        ]
        for pattern in patterns:
            message = re.sub(pattern, '[REDACTED]', message)
        return message